<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    Musaffar Patel
 * @copyright 2016-2025 Musaffar Patel
 * @license   LICENSE.txt
 */

use PrestaShop\Module\ProductPriceBySize\Controller\Admin\Product\ProductTabController;
use PrestaShop\Module\ProductPriceBySize\Controller\Front\Ppbs\ProductController;
use PrestaShop\Module\ProductPriceBySize\Schema\Schema;
use PrestaShop\Module\ProductPriceBySize\Service\BridgeService;
use PrestaShop\PrestaShop\Adapter\SymfonyContainer;

if (!defined('_PS_VERSION_')) {
    exit;
}

require_once __DIR__ . '/vendor/autoload.php';

class ProductPriceBySize extends Module
{
    public $moduleFile = __FILE__;

    public function __construct()
    {
        $this->name = 'productpricebysize';
        $this->tab = 'others';
        $this->version = '3.0.1';
        $this->author = 'Musaffar Patel';
        parent::__construct();
        $this->displayName = $this->l('Product Price By Size');
        $this->description = $this->l('Allow customers to buy products by area or area packs and calculate prices and weights dynamically');
        $this->module_key = 'a44fdab3786e7699a005cbaacc7f36f2';
        $this->confirmUninstall = $this->l('Are you sure you want to uninstall?');
        $this->file = __FILE__;
        $this->bootstrap = true;
        $this->ps_versions_compliancy = ['min' => '1.7.8', 'max' => _PS_VERSION_];
   }

    public function install()
    {
        if (!parent::install()
            || !$this->installModule()
            || !$this->registerHook('displayHeader')
            || !$this->registerHook('displayBackOfficeHeader')
            || !$this->registerHook('displayProductAdditionalInfo')
            || !$this->registerHook('actionProductPriceCalculation')
            || !$this->registerHook('displayCustomization')
            || !$this->registerHook('displayProductPriceBlock')
            || !$this->registerHook('displayAdminProductsExtra')
            || !$this->registerHook('actionProductAdd')
            || !$this->registerHook('actionProductUpdate')
            || !$this->registerHook('actionProductDelete')
        ) {
            return false;
        }

        return true;
    }

    private function installModule(): bool
    {
        $installSchema = new Schema();
        $installSchema->installSchema();
        $installSchema->installData();
        $installSchema->removeIndexFiles(_PS_MODULE_DIR_ . '/productpricebysize/config');

        return true;
    }

    public function uninstall(): bool
    {
        return parent::uninstall();
    }

    public function setMedia()
    {
        // (new ProductController($this))->setMedia();
    }

    public function hookDisplayHeader(array $params)
    {
        // $this->setMedia();
        $idProduct = (int) Tools::getValue('id_product');
        $factory = $this->get('prestashop.module.productpricebysize.factory.product_presenter_factory');
        $presenter = $factory->get($idProduct);
        if (empty($presenter)) {
            return;
        }
        $presenter->hookDisplayHeader($params, $idProduct);
    }

    public function hookDisplayBackOfficeHeader($params)
    {
        (new ProductTabController($this))->setMedia($params);
    }

    public function hookDisplayProductAdditionalInfo(array $params): string
    {
        // do not render when Prestashop refreshes this hook for example when attribute is changed
        if (Tools::getValue('action') !== 'refresh') {
            $idProduct = (int) $params['product']['id_product'];
            $factory = $this->get('prestashop.module.productpricebysize.factory.product_presenter_factory');
            $presenter = $factory->get($idProduct);
            if (!empty($presenter)) {
                return $presenter->renderHookDisplayProductAdditionalInfo($this->smarty, $presenter->present($idProduct));
            }
        }

        return '';
    }

    public function getContent()
    {
        $route = SymfonyContainer::getInstance()->get('router')->generate('productpricebysize_admin_config_dimensions');
        Tools::redirectAdmin($route);
    }

    public function hookActionProductPriceCalculation(array $params)
    {
        $factory = $this->get('prestashop.module.productpricebysize.factory.price_service_factory');
        $priceService = $factory->get((int) $params['id_product']);
        if (empty($priceService) || empty($params)) {
            return;
        }
        $params['pco_price_impact'] = BridgeService::getPcoPriceImpact($this->context, json_decode(json_encode($params), true), false);
        $params['price'] = $priceService->priceCalculation($params, null, false, true);
    }

    public function isUsingNewTranslationSystem(): bool
    {
        return true;
    }

    public function hookDisplayProductPriceBlock(array $params): string
    {
        return (new PrestaShop\Module\ProductPriceBySize\Controller\Front\Ppbs\CartController($this))->hookDisplayProductPriceBlock($params);
    }

    public function hookDisplayCustomization($params): string
    {
        $customization = $params['customization'] ?? '';
        $cartPresenter = $this->get('prestashop.module.productpricebysize.presenter.front.cart_presenter');

        return $cartPresenter->presentCustomizationString($customization['value'], false);
    }

    public function hookDisplayAdminProductsExtra(array $params): string
    {
        $idProduct = (int) $params['id_product'];
        $presenter = $this->get('prestashop.module.productpricebysize.presenter.admin.product.product_tab_presenter');
        $presented = $presenter->present($idProduct);

        return $this->get('twig')->render('@Modules/productpricebysize/views/templates/admin/product/hook/display_admin_products_extra.html.twig', $presented);
    }

    public function hookActionProductAdd(array $params): void
    {
        if (!empty($params['request'])) {
            $idProductOld = $params['id_product_old'];
            $idProduct = $params['id_product'];

            if ((int)$idProduct != (int)$idProductOld) {
                $productDeleteService = $this->get('prestashop.module.productpricebysize.admin.product_delete_service');
                $massAssignService = $this->get('prestashop.module.productpricebysize.admin.mass_assign_service');
                $productDeleteService->deleteSettings($idProduct);
                $massAssignService->duplicateProduct($idProductOld, $idProduct);
            }
        }
    }

    public function hookActionProductUpdate(array $params): void
    {
        $massAssignService = $this->get('prestashop.module.productpricebysize.admin.mass_assign_service');
        $massAssignService->setCustomizationFieldModuleFlag((int) $params['id_product']);
    }

    public function hookActionProductDelete($params): void
    {
        $productDeleteService = $this->get('prestashop.module.productpricebysize.admin.product_delete_service');
        $productDeleteService->deleteSettings((int) $params['id_product']);
    }
}
