<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    Musaffar Patel
 * @copyright 2016-2025 Musaffar Patel
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 */

declare(strict_types=1);

namespace PrestaShop\Module\ProductPriceBySize\Controller\Admin\Config;

use PrestaShop\Module\ProductPriceBySize\Grid\Filters\DimensionFilters;
use PrestaShop\PrestaShop\Core\Form\FormHandlerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

if (!defined('_PS_VERSION_')) {
    exit;
}

class DimensionsController extends BaseController
{
    public function indexAction(DimensionFilters $filters, Request $request): Response
    {
        $gridFactory = $this->get('productpricebysize.grid.factory.dimension');
        $grid = $gridFactory->getGrid($filters);

        return $this->render(
            '@Modules/productpricebysize/views/templates/admin/config/dimension/list.html.twig',
            [
                'enableSidebar' => true,
                'showContentHeader' => true,
                'layoutTitle' => $this->trans('Dimensions', 'Modules.Productpricebysize.Admin'),
                'layoutHeaderToolbarBtn' => [
                    'add' => [
                        'class' => 'btn-primary btn-group-add',
                        'href' => $this->generateUrl('productpricebysize_admin_config_dimensions_new', ['id_dimension' => 0]),
                        'desc' => $this->trans('New Dimension', 'Modules.Productpricebysize.Admin'),
                        'icon' => 'add_circle_outline',
                    ],
                ],
                'headerTabContent' => $this->renderHeaderTabs('tab_admin_config_dimensions'),
                'grid' => $this->presentGrid($grid),
            ]
        );
    }

    public function editAction(Request $request): Response
    {
        $this->get('prestashop.module.productpricebysize.form.config.dimension.data_provider')->setId((int) $request->get('id_dimension'));
        $formDataHandler = $this->get('prestashop.module.productpricebysize.form.config.dimension.data_handler');
        $form = $formDataHandler->getForm();
        $form->handleRequest($request);

        if ($form->isSubmitted()) {
            return $this->processAction($request, $formDataHandler);
        }

        return $this->render(
            '@Modules/productpricebysize/views/templates/admin/config/dimension/edit.html.twig',
            [
                'enableSidebar' => true,
                'showContentHeader' => true,
                'layoutTitle' => $this->trans('Dimension Configuration', 'Modules.Productpricebysize.Admin'),
                'headerTabContent' => $this->renderHeaderTabs('tab_admin_config_dimensions'),
                'form' => $form->createView(),
                'images' => !empty($form->getViewData()['images']) ? $form->getViewData()['images'] : [],
            ]
        );
    }

    public function deleteAction(Request $request): Response
    {
        $idDimension = (int) $request->get('id_dimension');
        if (empty($idDimension)) {
            $this->addFlash('error', $this->trans('Invalid Dimension Identifier provided', 'Modules.Productpricebysize.Admin'));
        } else {
            $dataProvider = $this->get('prestashop.module.productpricebysize.form.config.dimension.data_provider')->setId($idDimension);
            $dataProvider->delete();
        }

        return $this->redirectToRoute('productpricebysize_admin_config_dimensions');
    }

    public function processAction(Request $request, FormHandlerInterface $formDataHandler): Response
    {
        $form = $formDataHandler->getForm();
        $form->handleRequest($request);
        $data = $form->getData();

        if (empty($data['name'])) {
            $this->addFlash('error', $this->trans('Please enter a name for the dimension', 'Modules.Productpricebysize.Admin'));
        } else {
            $this->addFlash('success', $this->trans('Dimension saved successfully', 'Modules.Productpricebysize.Admin'));
        }
        $formDataHandler->save($data);

        return $this->redirectToRoute('productpricebysize_admin_config_dimensions');
    }

    public function deleteImageAction(Request $request): Response
    {
        $dataProvider = $this->get('prestashop.module.productpricebysize.form.config.dimension.data_provider')->setId((int) $request->get('id_dimension'));
        $dataProvider->deleteImage((int) $request->get('id_dimension'), (int) $request->get('id_lang'));
        $this->addFlash('success', $this->trans('Dimension image has been removed', 'Modules.Productpricebysize.Admin'));

        return $this->redirectToRoute('productpricebysize_admin_config_dimensions_edit', ['id_dimension' => (int) $request->get('id_dimension')]);
    }
}
