<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    Musaffar Patel
 * @copyright 2016-2025 Musaffar Patel
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 */

declare(strict_types=1);

namespace PrestaShop\Module\ProductPriceBySize\Controller\Admin\Product\Ppbs;

use PrestaShop\Module\ProductPriceBySize\Grid\Filters\AreaPriceFilters;
use PrestaShop\PrestaShop\Core\Form\FormHandlerInterface;
use Symfony\Component\Form\Form;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

if (!defined('_PS_VERSION_')) {
    exit;
}

class AreaPriceController extends BaseController
{
    public function indexAction(AreaPriceFilters $filters, Request $request): Response
    {
        $idProduct = (int) $request->get('id_product');
        $gridFactory = $this->get('productpricebysize.grid.factory.area_price');
        $grid = $gridFactory->getGrid($filters);

        return $this->render(
            '@Modules/productpricebysize/views/templates/admin/global/list.html.twig',
            [
                'enableSidebar' => true,
                'showContentHeader' => true,
                'layoutTitle' => $this->trans('Area Prices', 'Modules.Productpricebysize.Admin'),
                'layoutHeaderToolbarBtn' => [
                    'add' => [
                        'class' => 'btn-primary btn-group-add',
                        'href' => $this->generateUrl('productpricebysize_admin_product_ppbs_area_price_new', ['id_product' => $idProduct]),
                        'desc' => $this->trans('Add area based price', 'Modules.Productpricebysize.Admin'),
                        'icon' => 'add_circle_outline',
                    ],
                ],
                'headerTabContent' => $this->renderHeaderTabs('tab_admin_product_ppbs_area_price', $idProduct),
                'grid' => $this->presentGrid($grid),
                'parentProduct' => $this->getParentProductInfo($idProduct),
            ]
        );
    }

    public function editAction(Request $request): Response
    {
        $idProduct = (int) $request->get('id_product');
        $idAreaPrice = (int) $request->get('id_area_price');

        if (!empty($idAreaPrice)) {
            $productAreaPriceRepository = $this->get('prestashop.module.productpricebysize.repository.area_price_repository');
            $productAreaPrice = $productAreaPriceRepository->find($idAreaPrice);
            $idProduct = $productAreaPrice->getIdProduct();
        }

        $this->get('prestashop.module.productpricebysize.form.product.ppbs.area_price.data_provider')
            ->setIdProduct($idProduct)
            ->setId($idAreaPrice);

        $formDataHandler = $this->get('prestashop.module.productpricebysize.form.product.ppbs.area_price.data_handler');
        $form = $formDataHandler->getForm();
        $form->handleRequest($request);

        if ($form->isSubmitted()) {
            return $this->processAction($request, $form, $formDataHandler);
        }

        return $this->render(
            '@Modules/productpricebysize/views/templates/admin/global/edit.html.twig',
            [
                'enableSidebar' => true,
                'showContentHeader' => true,
                'layoutTitle' => $this->trans('Area Price Options', 'Modules.Productpricebysize.Admin'),
                'headerTabContent' => $this->renderHeaderTabs('tab_admin_product_ppbs_area_price', $idProduct),
                'form' => $form->createView(),
            ]
        );
    }

    public function processAction(Request $request, Form $form, FormHandlerInterface $formDataHandler): Response
    {
        $data = $form->getData();
        try {
            $result = $formDataHandler->save($data);
            $this->addFlash('success', $this->trans('Area based price saved successfully', 'Modules.Productpricebysize.Admin'));
        } catch (\Exception $e) {
            $this->addFlash('error', $this->trans('Error occurred while saving the area based price', 'Modules.Productpricebysize.Admin'));
        }

        return $this->redirectToRoute('productpricebysize_admin_product_ppbs_area_price', ['id_product' => $result['id_product']]);
    }

    public function deleteAction(Request $request): Response
    {
        $result = [];
        $idAreaPrice = (int) $request->get('id_area_price');

        if (!empty($idAreaPrice)) {
            $dataProvider = $this->get('prestashop.module.productpricebysize.form.product.ppbs.area_price.data_provider')->setId($idAreaPrice);
            $result = $dataProvider->delete();
            $this->addFlash('success', $this->trans('Area based price saved successfully', 'Modules.Productpricebysize.Admin'));
        } else {
            $this->addFlash('error', $this->trans('Error deleting area price', 'Modules.Productpricebysize.Admin'));
        }

        return $this->redirectToRoute('productpricebysize_admin_product_ppbs_area_price', ['id_product' => $result['id_product']]);
    }
}
