<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    Musaffar Patel
 * @copyright 2016-2025 Musaffar Patel
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 */

declare(strict_types=1);

namespace PrestaShop\Module\ProductPriceBySize\Form\Config;

if (!defined('_PS_VERSION_')) {
    exit;
}

use Doctrine\ORM\EntityManagerInterface;
use PrestaShop\Module\ProductPriceBySize\Entity\PpbsDimension;
use PrestaShop\Module\ProductPriceBySize\Entity\PpbsDimensionLang;
use PrestaShop\Module\ProductPriceBySize\Repository\DimensionRepository;
use PrestaShop\Module\ProductPriceBySize\Service\AssetsService;
use PrestaShop\Module\ProductPriceBySize\Service\ImageUploaderService;
use PrestaShop\PrestaShop\Core\Form\FormDataProviderInterface;
use Symfony\Component\HttpFoundation\File\UploadedFile;

/**
 * Provider ir responsible for providing form data, in this case it's as simple as using configuration to do that
 */
class DimensionFormDataProvider implements FormDataProviderInterface
{
    protected $context;

    protected $id;

    protected $entityManager;

    protected $dimensionRepository;

    protected $languages = [];

    public function __construct(
        \Context $context,
        EntityManagerInterface $entityManager,
        DimensionRepository $dimensionRepository,
        array $languages) {
        $this->context = $context;
        $this->entityManager = $entityManager;
        $this->dimensionRepository = $dimensionRepository;
        $this->languages = $languages;
    }

    public function setId(int $id): self
    {
        $this->id = $id;

        return $this;
    }

    public function getData(): array
    {
        if (empty($this->id)) {
            return [];
        }

        $dimension = $this->dimensionRepository->getOne($this->id);
        if (empty($dimension)) {
            return [];
        }

        $data['id'] = $dimension['id'];
        $data['name'] = $dimension['name'];
        foreach ($dimension['lang'] as $lang) {
            $data['display_name'][$lang['idLang']] = $lang['displayName'];
        }

        $images = [];
        if (!empty($dimension['lang'])) {
            foreach ($dimension['lang'] as $lang) {
                if (!empty($lang['image'])) {
                    $language = new \Language($lang['idLang']);

                    $images[] = [
                        'idDimension' => $dimension['id'],
                        'idLang' => $lang['idLang'],
                        'thumbnail' => ImageUploaderService::getUrlToImages('small') . $lang['image'],
                        'large' => ImageUploaderService::getUrlToImages('large') . $lang['image'],
                        'languageIso' => $language->iso_code,
                    ];
                }
            }
        }
        $data['images'] = $images;

        return $data;
    }

    public function setData(array $data): array
    {
        // Update existing Dimension
        if (!empty($data['id'])) {
            $dimension = $this->dimensionRepository->find((int) $data['id']);
            foreach ($dimension->getDimensionLangs() as $existingDimensionLang) {
                $this->entityManager->remove($existingDimensionLang);  // Remove each dimensionLang
            }
            $dimension->getDimensionLangs()->clear();
            $this->entityManager->flush();
        } else {
            $dimension = new PpbsDimension();
        }

        $dimension->setName($data['name']);

        foreach ($this->languages as $language) {
            $dimensionLang = new PpbsDimensionLang();
            $dimensionLang->setIdLang((int) $language['id_lang']);
            $displayName = !empty($data['display_name'][$language['id_lang']]) ? $data['display_name'][$language['id_lang']] : '';
            $dimensionLang->setDisplayName($displayName);
            $dimensionLang->setDimension($dimension);

            if (!empty($data['id'])) {
                $dimension->getDimensionLangs()->add($dimensionLang);
            } else {
                $dimension->setDimensionLangs([$dimensionLang]);
            }

            // Image
            $imageUploaderService = new ImageUploaderService($this->context);
            if (($data['image_' . $language['id_lang']] ?? null) instanceof UploadedFile) {
                $imageField = $data['image_' . $language['id_lang']];
                $imageUploadResult = $imageUploaderService->upload($imageField, 'dimension');
                $value = $imageUploadResult['filename'];
                $dimensionLang->setImage($value);
            }

            $this->entityManager->persist($dimensionLang);
        }
        $this->entityManager->persist($dimension);
        $this->entityManager->flush();

        return [];
    }

    public function deleteImage(int $idDimension, int $idLang): bool
    {
        $existingDimensionLang = $this->dimensionRepository->getDimensionLang($idDimension, $idLang);
        if (!empty($existingDimensionLang['image'])) {
            try {
                unlink(AssetsService::getThumbnailImagesPath() . $existingDimensionLang['image']);
                unlink(AssetsService::getLargeImagesPath() . $existingDimensionLang['image']);
            } catch (\Exception $e) {
            }
            $this->dimensionRepository->deleteDimensionLang($idDimension, $idLang);
        }

        return true;
    }

    public function delete(): bool
    {
        $dimension = $this->dimensionRepository->getOne($this->id);

        // remove dimension images from disk
        if (!empty($dimension['lang'])) {
            foreach ($dimension['lang'] as $lang) {
                if (!empty($lang['image'])) {
                    if (file_exists(AssetsService::getThumbnailImagesPath() . $lang['image'])) {
                        unlink(AssetsService::getThumbnailImagesPath() . $lang['image']);
                    }
                    if (file_exists(AssetsService::getLargeImagesPath() . $lang['image'])) {
                        unlink(AssetsService::getLargeImagesPath() . $lang['image']);
                    }
                }
            }
        }

        $this->dimensionRepository->softDelete($this->id);

        return true;
    }
}
