<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    Musaffar Patel
 * @copyright 2016-2025 Musaffar Patel
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 */

declare(strict_types=1);

namespace PrestaShop\Module\ProductPriceBySize\Form\Config;

if (!defined('_PS_VERSION_')) {
    exit;
}

use Doctrine\ORM\EntityManagerInterface;
use PrestaShop\Module\ProductPriceBySize\Entity\PpbsEquationTemplate;
use PrestaShop\Module\ProductPriceBySize\Entity\PpbsProductEquation;
use PrestaShop\Module\ProductPriceBySize\Repository\EquationTemplateRepository;
use PrestaShop\Module\ProductPriceBySize\Repository\EquationTypeRepository;
use PrestaShop\Module\ProductPriceBySize\Repository\ProductEquationRepository;
use PrestaShop\Module\ProductPriceBySize\Repository\ProductRepository;
use PrestaShop\PrestaShop\Core\Form\FormDataProviderInterface;

/**
 * Provider ir responsible for providing form data, in this case it's as simple as using configuration to do that
 */
class EquationTemplateFormDataProvider implements FormDataProviderInterface
{
    protected $context;

    protected $id;

    protected $equationTemplateRepository;

    protected $entityManager;

    protected $equationTypeRepository;

    protected $productEquationRepository;

    protected $productRepository;

    protected $languages = [];

    protected $idProduct = 0;

    protected $idProductAttribute = 0;

    protected $idEquationType = 0;

    protected $source = 'global';

    protected $equationType;

    public function __construct(
        \Context $context,
        EntityManagerInterface $entityManager,
        EquationTemplateRepository $equationTemplateRepository,
        EquationTypeRepository $equationTypeRepository,
        ProductEquationRepository $productEquationRepository,
        ProductRepository $productRepository) {
        $this->context = $context;
        $this->equationTemplateRepository = $equationTemplateRepository;
        $this->entityManager = $entityManager;
        $this->equationTypeRepository = $equationTypeRepository;
        $this->productEquationRepository = $productEquationRepository;
        $this->productRepository = $productRepository;
    }

    public function setId(int $id): self
    {
        $this->id = $id;

        return $this;
    }

    public function setIdProduct(int $idProduct): self
    {
        $this->idProduct = $idProduct;

        return $this;
    }

    public function setIdProductAttribute(int $idProductAttribute): self
    {
        $this->idProductAttribute = $idProductAttribute;

        return $this;
    }

    public function setIdEquationType(int $idEquationType): self
    {
        $this->idEquationType = $idEquationType;

        return $this;
    }

    public function setSource(string $source): self
    {
        $this->source = $source;

        return $this;
    }

    public function getData(): array
    {
        $ppbsEquationTemplate = null;

        if ($this->id > 0) {
            $ppbsEquationTemplate = $this->equationTemplateRepository->find($this->id);
            $data['id_equation_template'] = $ppbsEquationTemplate->getId();
            $data['equation'] = $ppbsEquationTemplate->getEquation();
            $data['id_equation_type'] = $ppbsEquationTemplate->getEquationType()->getId();
        } else {
            if ($this->idProduct > 0 || $this->idProductAttribute > 0) {
                $productEquation = $this->productEquationRepository->findOneBy([
                    'idProduct' => $this->idProduct,
                    'idProductAttribute' => $this->idProductAttribute,
                    'idEquationType' => $this->idEquationType,
                ]);
                if (!empty($productEquation)) {
                    $ppbsEquationTemplate = $productEquation->getEquationTemplate();
                }
            }
        }

        $data['id_product'] = $this->idProduct;
        $data['id_product_attribute'] = $this->idProductAttribute;

        if (!empty($ppbsEquationTemplate)) {
            $data['id_equation_template'] = $ppbsEquationTemplate->getId();
            $data['is_global'] = $ppbsEquationTemplate->getIsGlobal();
            $data['name'] = $ppbsEquationTemplate->getName();
            $data['equation'] = $ppbsEquationTemplate->getEquation();
            $data['id_equation_type'] = $ppbsEquationTemplate->getEquationType()->getId();
        } else {
            $data['id_equation_template'] = 0;
            $data['name'] = '';
            $data['equation'] = '';
            $data['id_equation_type'] = $this->idEquationType;

            if ($this->source == 'product') {
                $data['is_global'] = 0;
            } else {
                $data['is_global'] = 1;
            }
        }
        $data['source'] = $this->source;

        return $data;
    }

    public function setDataForProduct(array $data): array
    {
        $idProduct = (int) $data['id_product'];
        $idProductAttribute = (int) $data['id_product_attribute'];

        if ($idProduct > 0 || $idProductAttribute > 0) {
            if ($data['save_method'] == 'custom') {
                $data['is_global'] = 0;
            } else {
                $data['is_global'] = 1;
            }

            $ppbsProductEquation = $this->productEquationRepository->findOneBy([
                'idProduct' => $data['id_product'],
                'idProductAttribute' => $data['id_product_attribute'],
                'idEquationType' => $data['id_equation_type'],
            ]);

            // delete any specific (non-global) equation template associated with this product / product attribute
            if (!empty($ppbsProductEquation)) {
                if ($ppbsProductEquation->getEquationTemplate()->getIsGlobal() == 0) {
                    $this->entityManager->remove($ppbsProductEquation->getEquationTemplate());
                    $this->entityManager->flush();
                }
                $this->entityManager->remove($ppbsProductEquation);
                $this->entityManager->flush();
            }

            $ppbsEquationType = $this->equationTypeRepository->find($data['id_equation_type']);

            if ($data['save_method'] == 'custom' || (int) $data['id_equation_template'] == 0) {
                $ppbsEquationTemplate = new PpbsEquationTemplate();
                $ppbsEquationTemplate->setEquation($data['equation']);
                $ppbsEquationTemplate->setEquationType($ppbsEquationType);
                $ppbsEquationTemplate->setIsGlobal(!empty($data['is_global']) && $data['is_global'] == 1);
                if ($data['save_method'] == 'custom') {
                    $ppbsEquationTemplate->setName('');
                } else {
                    $ppbsEquationTemplate->setName($data['name']);
                }
                $this->entityManager->persist($ppbsEquationTemplate);
                $this->entityManager->flush();
            } else {
                $ppbsEquationTemplate = $this->equationTemplateRepository->find((int) $data['id_equation_template']);
            }

            $ppbsProductEquation = new PpbsProductEquation();
            $ppbsProductEquation->setProduct($this->productRepository->find($data['id_product']));
            $ppbsProductEquation->setIdProductAttribute((int) $data['id_product_attribute']);
            $ppbsProductEquation->setEquationTemplate($ppbsEquationTemplate);
            $this->entityManager->persist($ppbsProductEquation);
            $this->entityManager->flush();
        }

        return [];
    }

    public function setData(array $data): array
    {
        if ($data['source'] == 'product') {
            return $this->setDataForProduct($data);
        }

        $ppbsEquationType = $this->equationTypeRepository->find($data['id_equation_type']);

        if ((int) $data['id_equation_template'] > 0) {
            $ppbsEquationTemplate = $this->equationTemplateRepository->find((int) $data['id_equation_template']);
            $this->id = (int) $data['id_equation_template'];
        }

        if (empty($ppbsEquationTemplate)) {
            $ppbsEquationTemplate = new PpbsEquationTemplate();
            $this->id = null;
        }

        // when updating from global config, we're also updating the equation template itself.  When updating it from a specific product
        // we're only assigning the template to the product, not updating the template#'
        // or we're creating a new template (when id_equation_template == 0)
        if ($data['source'] == 'global' || ($data['source'] == 'product' && (int) $data['id_equation_template'] == 0)) {
            $ppbsEquationTemplate->setEquation($data['equation']);
            $ppbsEquationTemplate->setEquationType($ppbsEquationType);
            $ppbsEquationTemplate->setIsGlobal(!empty($data['is_global']) && $data['is_global'] == 1);
            $ppbsEquationTemplate->setName($data['name']);
        }

        $this->entityManager->persist($ppbsEquationTemplate);
        $this->entityManager->flush();

        if ($this->idProduct > 0 || $this->idProductAttribute > 0) {
            $ppbsProductEquation = $this->productEquationRepository->findOneBy([
                'idProduct' => $data['id_product'],
                'idProductAttribute' => $data['id_product_attribute'],
                'idEquationType' => $data['id_equation_type'],
            ]);

            if (empty($ppbsProductEquation)) {
                $ppbsProductEquation = new PpbsProductEquation();
                $ppbsProductEquation->setProduct($this->productRepository->find($data['id_product']));
                $ppbsProductEquation->setIdProductAttribute((int) $data['id_product_attribute']);
            }

            $ppbsProductEquation->setEquationTemplate($ppbsEquationTemplate);
            $this->entityManager->persist($ppbsProductEquation);
            $this->entityManager->flush();
        }

        return [];
    }

    public function delete(): void
    {
        $ppbsEquationTemplate = $this->equationTemplateRepository->find($this->id);
        $ppbsProductEquations = $this->productEquationRepository->findBy([
            'idEquationTemplate' => $this->id,
        ]);
        foreach ($ppbsProductEquations as $ppbsProductEquation) {
            $this->entityManager->remove($ppbsProductEquation);
        }
        $this->entityManager->flush();

        if ($ppbsEquationTemplate !== null) {
            $this->entityManager->remove($ppbsEquationTemplate);
            $this->entityManager->flush();
        }
    }

    public function deleteByProduct(int $idProduct, int $idProductAttribute, int $idEquationType): void
    {
        $ppbsProductEquation = $this->productEquationRepository->findOneBy([
            'idProduct' => $idProduct,
            'idProductAttribute' => $idProductAttribute,
            'idEquationType' => $idEquationType,
        ]);

        if (!empty($ppbsProductEquation)) {
            $ppbsEquationTemplate = $ppbsProductEquation->getEquationTemplate();
            $this->entityManager->remove($ppbsProductEquation);
            $this->entityManager->flush();

            if ($ppbsEquationTemplate->getIsGlobal() == 0) {
                $this->entityManager->remove($ppbsEquationTemplate);
                $this->entityManager->flush();
            }
        }
    }
}
