<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    Musaffar Patel
 * @copyright 2016-2025 Musaffar Patel
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 */

declare(strict_types=1);

namespace PrestaShop\Module\ProductPriceBySize\Form\Config;

if (!defined('_PS_VERSION_')) {
    exit;
}

use Doctrine\ORM\EntityManagerInterface;
use Doctrine\Common\Collections\ArrayCollection;
use PrestaShop\Module\ProductPriceBySize\Entity\PpbsUnit;
use PrestaShop\Module\ProductPriceBySize\Entity\PpbsUnitLang;
use PrestaShop\Module\ProductPriceBySize\Repository\UnitRepository;
use PrestaShop\PrestaShop\Core\Form\FormDataProviderInterface;

/**
 * Provider ir responsible for providing form data, in this case it's as simple as using configuration to do that
 */
class UnitFormDataProvider implements FormDataProviderInterface
{
    protected $context;

    protected $id;

    protected $entityManager;

    protected $unitRepository;

    protected $languages = [];

    public function __construct(
        \Context $context,
        EntityManagerInterface $entityManager,
        UnitRepository $unitRepository,
        array $languages) {
        $this->context = $context;
        $this->entityManager = $entityManager;
        $this->unitRepository = $unitRepository;
        $this->languages = $languages;
    }

    public function setId(int $id): self
    {
        $this->id = $id;

        return $this;
    }

    public function getData(): array
    {
        if (empty($this->id)) {
            return [];
        }
        $unit = $this->unitRepository->getOne($this->id);
        if (empty($unit)) {
            return [];
        }

        $data['id'] = $unit['id'];
        $data['name'] = $unit['name'];
        $data['conversionFactor'] = $unit['conversionFactor'];
        foreach ($unit['lang'] as $lang) {
            $data['displayName'][$lang['idLang']] = $lang['displayName'];
        }

        return $data;
    }

    public function setData(array $data): array
    {
        // Update existing Unit
        if (!empty($data['id'])) {
            $unit = $this->unitRepository->find((int) $data['id']);
            foreach ($unit->getLangs() as $existingLang) {
                $this->entityManager->remove($existingLang);  // Remove each unit lang entry
            }
            $unit->getLangs()->clear();
            $this->entityManager->flush();
        } else {
            $unit = new PpbsUnit();
        }

        $unit->setName($data['name']);
        $unit->setConversionFactor((float) $data['conversionFactor']);

        foreach ($this->languages as $language) {
            $unitLang = new PpbsUnitLang();
            $unitLang->setIdLang((int) $language['id_lang']);
            $displayName = !empty($data['displayName'][$language['id_lang']]) ? $data['displayName'][$language['id_lang']] : '';
            $unitLang->setDisplayName($displayName);
            $unitLang->setUnit($unit);

            if (!empty($data['id'])) {
                $unit->getLangs()->add($unitLang);
            } else {
                $unit->setLangs(new ArrayCollection([$unitLang]));
            }
            $this->entityManager->persist($unitLang);
        }
        $this->entityManager->persist($unit);
        $this->entityManager->flush();

        return [];
    }

    public function delete(): bool
    {
        $this->unitRepository->softDelete($this->id);

        return true;
    }
}
