<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    Musaffar Patel
 * @copyright 2016-2025 Musaffar Patel
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 */

declare(strict_types=1);

namespace PrestaShop\Module\ProductPriceBySize\Form\Product\Pap;

if (!defined('_PS_VERSION_')) {
    exit;
}

use Doctrine\ORM\EntityManagerInterface;
use PrestaShop\Module\ProductPriceBySize\Entity\PpbsProduct;
use PrestaShop\Module\ProductPriceBySize\Entity\PpbsProductPack;
use PrestaShop\Module\ProductPriceBySize\Entity\PpbsProductUnit;
use PrestaShop\Module\ProductPriceBySize\Entity\PpbsProductWastage;
use PrestaShop\Module\ProductPriceBySize\Enum\CalculationType;
use PrestaShop\Module\ProductPriceBySize\Enum\PackType;
use PrestaShop\Module\ProductPriceBySize\Repository\ProductPackRepository;
use PrestaShop\Module\ProductPriceBySize\Repository\ProductRepository;
use PrestaShop\Module\ProductPriceBySize\Repository\UnitRepository;
use PrestaShop\Module\ProductPriceBySize\Repository\WastageRepository;
use PrestaShop\PrestaShop\Core\Form\FormDataProviderInterface;

/**
 * Provider ir responsible for providing form data, in this case it's as simple as using configuration to do that
 */
class PackFormDataProvider implements FormDataProviderInterface
{
    protected $context;

    protected $idProduct;

    protected $idProductAttribute;

    protected $entityManager;

    protected $productRepository;

    protected $productPackRepository;

    protected $unitRepository;

    protected $wastageRepository;

    protected $languages = [];

    public function __construct(
        \Context $context,
        EntityManagerInterface $entityManager,
        ProductRepository $productRepository,
        ProductPackRepository $productPackRepository,
        UnitRepository $unitRepository,
        WastageRepository $wastageRepository,
        array $languages) {
        $this->context = $context;
        $this->entityManager = $entityManager;
        $this->productRepository = $productRepository;
        $this->productPackRepository = $productPackRepository;
        $this->unitRepository = $unitRepository;
        $this->wastageRepository = $wastageRepository;
        $this->languages = $languages;
    }

    public function setIdProduct(int $idProduct): self
    {
        $this->idProduct = $idProduct;

        return $this;
    }

    public function setIdProductAttribute(int $idProductAttribute): self
    {
        $this->idProductAttribute = $idProductAttribute;

        return $this;
    }

    public function getData(): array
    {
        $data = [];
        $productUnitChoices = [];
        $productWastageChoices = [];
        $data['id_product'] = (int) $this->idProduct;
        $data['id_product_attribute'] = (int) $this->idProductAttribute;

        if (empty($this->idProduct)) {
            return $data;
        }

        $product = $this->productRepository->find($this->idProduct);
        $productPack = $this->productPackRepository->findOneBy([
            'idProduct' => $data['id_product'],
            'idProductAttribute' => $data['id_product_attribute'],
        ]);

        // Are we displaying the product settings or settings for a specific attribute combination?
        if ($this->idProductAttribute == 0) {
            if (!empty($product)) {
                $data['is_enabled'] = $product->getIsEnabled();
                $data['id_unit'] = $product->getUnit()->getId();
            }
        }
        $data['id_product_attribute'] = $this->idProductAttribute;

        if (!empty($product)) {
            $productUnits = $product->getProductUnits();
            foreach ($productUnits as $productUnit) {
                $productUnitChoices[] = $productUnit->getUnit()->getId();
            }

            $productWastages = $product->getWastageOptions();
            foreach ($productWastages as $productWastage) {
                $productWastageChoices[] = $productWastage->getWastage()->getId();
            }
        }

        if (!empty($productPack)) {
            $data['pack_area'] = $productPack->getPackArea();
            $data['area_price'] = $productPack->getAreaPrice();
            $data['pack_type'] = $product->getPapType();
            $data['roll_height'] = $productPack->getRollHeight();
            $data['roll_width'] = $productPack->getRollWidth();
            $data['pattern_repeat'] = $productPack->getPatternRepeat();
            $data['coverage'] = $productPack->getCoverage();
            $data['product_unit'] = $productUnitChoices;
            $data['product_wastage'] = $productWastageChoices;

            switch ($product->getPapType()) {
                case 'normal':
                    $data['pack_type'] = PackType::NORMAL;
                    break;
                case 'rolls':
                    $data['pack_type'] = PackType::ROLLS;
                    break;
                case 'paint':
                    $data['pack_type'] = PackType::PAINT;
                    break;
            }

            if ($product->getIsDynamicPrice()) {
                $data['calculation_type'] = CalculationType::DYNAMIC;
            } else {
                $data['calculation_type'] = CalculationType::NORMAL;
            }
        }

        return $data;
    }

    public function setData(array $data): array
    {
        if (empty($data['id_product'])) {
            return [];
        }

        $idProduct = (int) $data['id_product'];

        $product = $this->productRepository->find((int) $data['id_product']);
        $productPack = $this->productPackRepository->findOneBy([
            'idProduct' => (int) $data['id_product'],
            'idProductAttribute' => (int) $data['id_product_attribute'],
        ]);

        // Are we updating the product settings or settings for a specific attribute combination?
        if ($this->idProductAttribute == 0) {
            $unit = $this->unitRepository->find((int) $data['id_unit']);
            if (empty($product)) {
                $product = new PpbsProduct();
                $product->setId((int) $data['id_product']);
            }
            $product->setType('pap');
            $product->setIsEnabled(true);
            $product->setUnit($unit);
            $product->setPapType($data['pack_type']);
            $product->setIsDynamicPrice((bool) $data['calculation_type']);

            $this->entityManager->persist($product);
            $this->entityManager->flush();

            // Assign product units
            $this->productRepository->deleteProductUnits($idProduct);
            if (!empty($data['product_unit'])) {
                foreach ($data['product_unit'] as $idUnit) {
                    $unit = $this->unitRepository->find($idUnit);
                    $ppbsProductUnit = new PpbsProductUnit();
                    $ppbsProductUnit->setProduct($product);
                    $ppbsProductUnit->setUnit($unit);
                    $ppbsProductUnit->setIsDefault(boolval($data['id_unit'] === $idUnit));
                    $this->entityManager->persist($ppbsProductUnit);
                }
                $this->entityManager->flush();
            }

            $this->productRepository->deleteProductWastage($idProduct);
            if (!empty($data['product_wastage'])) {
                foreach ($data['product_wastage'] as $idWastage) {
                    $wastage = $this->wastageRepository->find($idWastage);
                    $ppbsProductWastage = new PpbsProductWastage();
                    $ppbsProductWastage->setProduct($product);
                    $ppbsProductWastage->setWastage($wastage);
                    $this->entityManager->persist($ppbsProductWastage);
                }
                $this->entityManager->flush();
            }
        }

        if (empty($productPack)) {
            $productPack = new PpbsProductPack();
            $productPack->setProduct($product);
            $productPack->setIdProductAttribute(0);
        }

        $productPack->setIdProductAttribute((int) $data['id_product_attribute']);
        $productPack->setPackArea(!empty($data['pack_area']) ? (float) $data['pack_area'] : 0);
        $productPack->setAreaPrice(!empty($data['area_price']) ? (float) $data['area_price'] : 0);
        $productPack->setRollHeight(!empty($data['roll_height']) ? (float) $data['roll_height'] : 0);
        $productPack->setRollWidth(!empty($data['roll_width']) ? (float) $data['roll_width'] : 0);
        $productPack->setPatternRepeat(!empty($data['pattern_repeat']) ? (float) $data['pattern_repeat'] : 0);
        $productPack->setCoverage(!empty($data['coverage']) ? (float) $data['coverage'] : 0);
        $this->entityManager->persist($productPack);
        $this->entityManager->flush();

        return [];
    }
}
