<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    Musaffar Patel
 * @copyright 2016-2025 Musaffar Patel
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 */

declare(strict_types=1);

namespace PrestaShop\Module\ProductPriceBySize\Form\Product\Ppbs;

if (!defined('_PS_VERSION_')) {
    exit;
}

use Doctrine\ORM\EntityManagerInterface;
use PrestaShop\Module\ProductPriceBySize\Entity\PpbsAreaPrice;
use PrestaShop\Module\ProductPriceBySize\Repository\AreaPriceRepository;
use PrestaShop\Module\ProductPriceBySize\Repository\ProductRepository;
use PrestaShop\PrestaShop\Core\Form\FormDataProviderInterface;

/**
 * Provider ir responsible for providing form data, in this case it's as simple as using configuration to do that
 */
class AreaPriceFormDataProvider implements FormDataProviderInterface
{
    protected $context;

    protected $idProduct;

    protected $idAreaPrice;

    protected $entityManager;

    protected $areaPriceRepository;

    protected $productRepository;

    public function __construct(
        \Context $context,
        EntityManagerInterface $entityManager,
        AreaPriceRepository $areaPriceRepository,
        ProductRepository $productRepository) {
        $this->context = $context;
        $this->entityManager = $entityManager;
        $this->areaPriceRepository = $areaPriceRepository;
        $this->productRepository = $productRepository;
    }

    public function setId(int $idAreaPrice): self
    {
        $this->idAreaPrice = $idAreaPrice;

        return $this;
    }

    public function setIdProduct(int $idProduct): self
    {
        $this->idProduct = $idProduct;

        return $this;
    }

    public function getData(): array
    {
        $data = [];
        if (!empty($this->idAreaPrice)) {
            $areaPrice = $this->areaPriceRepository->find($this->idAreaPrice);
            $data['id_area_price'] = $areaPrice->getId();
            $data['idProduct'] = $areaPrice->getProduct()->getId();
            $data['area_low'] = $areaPrice->getAreaLow();
            $data['area_high'] = $areaPrice->getAreaHigh();
            $data['price'] = $areaPrice->getPrice();
            $data['weight'] = $areaPrice->getWeight();
            $data['impact'] = $areaPrice->getImpact();
        } else {
            $data = [
                'idProduct' => $this->idProduct,
                'area_low' => 0,
                'area_high' => 0,
                'price' => 0,
                'weight' => 0,
                'impact' => '+',
            ];
        }

        return $data;
    }

    public function setData(array $data): array
    {
        if (empty($data['idProduct'])) {
            return [];
        }

        if (!empty($data['id_area_price'])) {
            $areaPrice = $this->areaPriceRepository->find((int) $data['id_area_price']);
        } else {
            $areaPrice = new PpbsAreaPrice();
            $areaPrice->setIdShop($this->context->shop->id);
            $areaPrice->setProduct($this->productRepository->find((int) $data['idProduct']));
        }

        $areaPrice->setAreaLow((float) $data['area_low']);
        $areaPrice->setAreaHigh((float) $data['area_high']);
        $areaPrice->setPrice((float) $data['price']);
        $areaPrice->setWeight((float) $data['weight']);
        $areaPrice->setImpact($data['impact']);
        $this->entityManager->persist($areaPrice);
        $this->entityManager->flush();

        return [
            'id_product' => $areaPrice->getProduct()->getId(),
        ];
    }

    public function delete(): array
    {
        $areaPrice = $this->areaPriceRepository->find($this->idAreaPrice);
        $this->entityManager->remove($areaPrice);
        $this->entityManager->flush();

        return [
            'id_product' => $areaPrice->getProduct()->getId(),
        ];
    }
}
