<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    Musaffar Patel
 * @copyright 2016-2025 Musaffar Patel
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 */

declare(strict_types=1);

namespace PrestaShop\Module\ProductPriceBySize\Form\Product\Ppbs;

if (!defined('_PS_VERSION_')) {
    exit;
}

use Doctrine\ORM\EntityManagerInterface;
use PrestaShop\Module\ProductPriceBySize\Entity\PpbsProduct;
use PrestaShop\Module\ProductPriceBySize\Entity\PpbsProductUnit;
use PrestaShop\Module\ProductPriceBySize\Repository\ProductRepository;
use PrestaShop\Module\ProductPriceBySize\Repository\UnitRepository;
use PrestaShop\PrestaShop\Core\Form\FormDataProviderInterface;

/**
 * Provider ir responsible for providing form data, in this case it's as simple as using configuration to do that
 */
class OptionsFormDataProvider implements FormDataProviderInterface
{
    protected $context;

    protected $idProduct;

    protected $entityManager;

    protected $productRepository;

    protected $unitRepository;

    protected $languages = [];

    public function __construct(
        \Context $context,
        EntityManagerInterface $entityManager,
        ProductRepository $productRepository,
        UnitRepository $unitRepository,
        array $languages) {
        $this->context = $context;
        $this->entityManager = $entityManager;
        $this->productRepository = $productRepository;
        $this->unitRepository = $unitRepository;
        $this->languages = $languages;
    }

    public function setIdProduct(int $idProduct): self
    {
        $this->idProduct = $idProduct;

        return $this;
    }

    public function getData(): array
    {
        $productUnitChoices = [];
        $product = $this->productRepository->find($this->idProduct);

        if (empty($product)) {
            return [
                'idProduct' => $this->idProduct,
            ];
        }
        $productUnits = $product->getProductUnits();
        foreach ($productUnits as $productUnit) {
            $productUnitChoices[] = $productUnit->getUnit()->getId();
        }

        $data = [
            'idProduct' => $this->idProduct,
            'is_enabled' => $product->getIsEnabled(),
            'id_unit' => $product->getIdUnit(),
            'is_front_conversion_enabled' => $product->getIsFrontConversionEnabled(),
            'front_conversion_operator' => $product->getFrontConversionOperator(),
            'front_conversion_value' => $product->getFrontConversionValue(),
            'min_total_area' => $product->getMinTotalArea(),
            'min_price' => $product->getMinPrice(),
            'setup_fee' => $product->getSetupFee(),
            'is_attribute_price_area_price' => $product->getIsAttributePriceAreaPrice(),
            'product_unit' => $productUnitChoices,
        ];

        return $data;
    }

    public function setData(array $data): array
    {
        if (empty($data['idProduct']) || empty($data['id_unit'])) {
            return [];
        }

        $idProduct = (int) $data['idProduct'];
        $ppbsProduct = $this->productRepository->find($idProduct);
        $ppbsUnit = $this->unitRepository->find((int) $data['id_unit']);

        if (empty($ppbsProduct)) {
            $ppbsProduct = new PpbsProduct();
            $ppbsProduct->setId($idProduct);
        }

        $ppbsProduct->setType('ppbs');
        $ppbsProduct->setIsEnabled(boolval($data['is_enabled']));
        $ppbsProduct->setUnit($ppbsUnit);
        /*$ppbsProduct->setIsFrontConversionEnabled(boolval($data['is_front_conversion_enabled']));
        $ppbsProduct->setFrontConversionOperator(pSQL($data['front_conversion_operator']));
        $ppbsProduct->setFrontConversionValue((float) $data['front_conversion_value']);*/

        $ppbsProduct->setIsFrontConversionEnabled(false);
        $ppbsProduct->setFrontConversionOperator('');
        $ppbsProduct->setFrontConversionValue(0);

        $ppbsProduct->setMinTotalArea((float) $data['min_total_area']);
        $ppbsProduct->setMinPrice((float) $data['min_price']);
        $ppbsProduct->setSetupFee((float) $data['setup_fee']);
        $ppbsProduct->setIsAttributePriceAreaPrice(boolval($data['is_attribute_price_area_price']));

        // Assign product units
        $this->productRepository->deleteProductUnits($idProduct);
        foreach ($data['product_unit'] as $idUnit) {
            $unit = $this->unitRepository->find($idUnit);
            $ppbsProductUnit = new PpbsProductUnit();
            $ppbsProductUnit->setProduct($ppbsProduct);
            $ppbsProductUnit->setUnit($unit);
            $ppbsProductUnit->setIsDefault(boolval($data['id_unit'] === $idUnit));
            $this->entityManager->persist($ppbsProductUnit);
        }

        $this->entityManager->persist($ppbsProduct);
        $this->entityManager->flush();

        return [];
    }
}
