<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    Musaffar Patel
 * @copyright 2016-2025 Musaffar Patel
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 */

declare(strict_types=1);

namespace PrestaShop\Module\ProductPriceBySize\Form\Product;

if (!defined('_PS_VERSION_')) {
    exit;
}

use Doctrine\ORM\EntityManagerInterface;
use PrestaShop\Module\ProductPriceBySize\Entity\PpbsProduct;
use PrestaShop\Module\ProductPriceBySize\Entity\PpbsProductField;
use PrestaShop\Module\ProductPriceBySize\Repository\DimensionRepository;
use PrestaShop\Module\ProductPriceBySize\Repository\FieldTypeRepository;
use PrestaShop\Module\ProductPriceBySize\Repository\ProductFieldRepository;
use PrestaShop\Module\ProductPriceBySize\Repository\ProductRepository;
use PrestaShop\Module\ProductPriceBySize\Repository\UnitRepository;
use PrestaShop\Module\ProductPriceBySize\Service\Admin\ProductDeleteService;
use PrestaShop\PrestaShop\Core\Form\FormDataProviderInterface;

/**
 * Provider ir responsible for providing form data, in this case it's as simple as using configuration to do that
 */
class ProductFieldFormDataProvider implements FormDataProviderInterface
{
    protected $context;

    protected $id = 0;

    protected $idProduct = 0;

    protected $entityManager;

    protected $productRepository;

    protected $productFieldRepository;

    protected $dimensionRepository;

    protected $fieldTypeRepository;

    protected $unitRepository;

    protected $productDeleteService;

    public function __construct(
        \Context $context,
        EntityManagerInterface $entityManager,
        ProductRepository $productRepository,
        ProductFieldRepository $productFieldRepository,
        DimensionRepository $dimensionRepository,
        FieldTypeRepository $fieldTypeRepository,
        UnitRepository $unitRepository,
        ProductDeleteService $productDeleteService) {
        $this->context = $context;
        $this->entityManager = $entityManager;
        $this->productRepository = $productRepository;
        $this->productFieldRepository = $productFieldRepository;
        $this->dimensionRepository = $dimensionRepository;
        $this->fieldTypeRepository = $fieldTypeRepository;
        $this->unitRepository = $unitRepository;
        $this->productDeleteService = $productDeleteService;
    }

    public function setId(int $id): self
    {
        $this->id = $id;

        return $this;
    }

    public function setIdProduct(int $idProduct): self
    {
        $this->idProduct = $idProduct;

        return $this;
    }

    public function getData(): array
    {
        $data = [];
        if ($this->id > 0) {
            $productField = $this->productFieldRepository->find($this->id);
            $data['id_product'] = $productField->getIdProduct();
            $data['id_product_field'] = $this->id;
            $data['id_dimension'] = $productField->getIdDimension();
            $data['id_unit'] = $productField->getIdUnit();
            $data['id_field_type'] = $productField->getIdFieldType();
            $data['min'] = $productField->getMin();
            $data['max'] = $productField->getMax();
            $data['step'] = $productField->getStep();
            $data['ratio'] = $productField->getRatio();
            $data['default_value'] = $productField->getDefaultValue();
            $data['decimals'] = $productField->getDecimals();
            $data['position'] = $productField->getPosition();
        } else {
            $data['id_product_field'] = 0;
            $data['id_product'] = $this->idProduct;
            $data['id_dimension'] = 0;
            $data['id_unit'] = 0;
            $data['field_type'] = 0;
            $data['min'] = 0;
            $data['max'] = 0;
            $data['step'] = 0;
            $data['ratio'] = 0;
            $data['default_value'] = null;
            $data['decimals'] = 0;
            $data['position'] = 0;
        }

        return $data;
    }

    public function setData(array $data): array
    {
        $idProductField = (int) $data['id_product_field'];
        $productField = $this->productFieldRepository->find($this->id);
        $ppbsProduct = $this->productRepository->find($data['id_product']);

        $unit = $this->unitRepository->find((int) $data['id_unit']);
        $dimension = $this->dimensionRepository->find((int) $data['id_dimension']);
        $fieldType = $this->fieldTypeRepository->find((int) $data['id_field_type']);

        if (empty($productField)) {
            $productField = new PpbsProductField();
            $productField->setIdProduct($this->idProduct);
            $productField->setPosition($this->productFieldRepository->getNewPosition($productField->getIdProduct()));
            $productField->setProduct($ppbsProduct);
        }

        $productField->setDimension($dimension);
        $productField->setFieldType($fieldType);
        $productField->setUnit($unit);
        $productField->setMin($data['min']);
        $productField->setMax($data['max']);
        $productField->setStep($data['step']);
        $productField->setRatio($data['ratio']);
        $productField->setDefaultValue($data['default_value']);
        $productField->setDecimals((int) $data['decimals']);
        $this->entityManager->persist($productField);
        $this->entityManager->flush();

        return [
            'idProductField' => $productField->getId(),
            'idProduct' => $productField->getIdProduct(),
            'fieldType' => $fieldType->getName(),
        ];
    }

    public function delete(): array
    {
        $productField = $this->productFieldRepository->find($this->id);

        return [
            'idProduct' => $this->productDeleteService->deleteProductField($this->id)['idProduct'],
            'type' => $productField->getProduct()->getType(),
        ];
    }

    public function updatePositions(array $positions, int $idProduct): PpbsProduct
    {
        $product = $this->productRepository->find($idProduct);
        $i = 0;
        foreach ($positions as $position) {
            $productField = $this->productFieldRepository->find((int) $position['rowId']);
            if (!empty($productField)) {
                $productField->setPosition($i);
                $this->entityManager->persist($productField);
                ++$i;
            }
        }
        $this->entityManager->flush();

        return $product;
    }
}
