<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    Musaffar Patel
 * @copyright 2016-2025 Musaffar Patel
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 */

declare(strict_types=1);

namespace PrestaShop\Module\ProductPriceBySize\Form\Product;

if (!defined('_PS_VERSION_')) {
    exit;
}

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\EntityManagerInterface;
use PrestaShop\Module\ProductPriceBySize\Entity\PpbsProductFieldValue;
use PrestaShop\Module\ProductPriceBySize\Entity\PpbsProductFieldValueLang;
use PrestaShop\Module\ProductPriceBySize\Repository\ProductFieldRepository;
use PrestaShop\Module\ProductPriceBySize\Repository\ProductFieldValueRepository;
use PrestaShop\PrestaShop\Core\Form\FormDataProviderInterface;

/**
 * Provider ir responsible for providing form data, in this case it's as simple as using configuration to do that
 */
class ProductFieldValueFormDataProvider implements FormDataProviderInterface
{
    protected $context;

    protected $id = 0;

    protected $idProductField = 0;

    protected $entityManager;

    protected $productFieldRepository;

    protected $productFieldValueRepository;

    protected $languages = [];

    public function __construct(
        \Context $context,
        EntityManagerInterface $entityManager,
        ProductFieldRepository $productFieldRepository,
        ProductFieldValueRepository $productFieldValueRepository,
        array $languages) {
        $this->context = $context;
        $this->entityManager = $entityManager;
        $this->productFieldRepository = $productFieldRepository;
        $this->productFieldValueRepository = $productFieldValueRepository;
        $this->languages = $languages;
    }

    public function setId(int $id): self
    {
        $this->id = $id;

        return $this;
    }

    public function setIdProductField(int $idProductField): self
    {
        $this->idProductField = $idProductField;

        return $this;
    }

    public function getData(): array
    {
        $data = [];
        $data['id_product_field_value'] = $this->id;

        if ($this->id > 0) {
            $productFieldValue = $this->productFieldValueRepository->find($this->id);
            $data['id_product_field_value'] = $productFieldValue->getId();
            $data['value'] = $productFieldValue->getValue();
            foreach ($productFieldValue->getLangs() as $lang) {
                $data['text'][$lang->getIdLang()] = $lang->getText();
            }
        } else {
            $data['id_product_field'] = $this->idProductField;
            $data['value'] = 0;
        }

        return $data;
    }

    private function deleteProductFieldValueLang(PpbsProductFieldValue $productFieldValue): void
    {
        foreach ($productFieldValue->getLangs() as $lang) {
            $this->entityManager->remove($lang);  // Remove each dimensionLang
        }
        $productFieldValue->getLangs()->clear();
        $this->entityManager->flush();
    }

    public function setData(array $data): PpbsProductFieldValue
    {
        if (!empty($data['id_product_field_value'])) {
            $productFieldValue = $this->productFieldValueRepository->find((int) $data['id_product_field_value']);
            $this->deleteProductFieldValueLang($productFieldValue);
        } else {
            $productFieldValue = new PpbsProductFieldValue();
            $productField = $this->productFieldRepository->find($data['id_product_field']);
            $productFieldValue->setProductField($productField);
            $productFieldValue->setPosition($this->productFieldValueRepository->getNewPosition($productField->getId()));
        }

        $productFieldValue->setValue($data['value']);

        foreach ($this->languages as $language) {
            $valueLang = new PpbsProductFieldValueLang();
            $valueLang->setIdLang((int) $language['id_lang']);
            $optionText = !empty($data['text'][$language['id_lang']]) ? $data['text'][$language['id_lang']] : '';
            $valueLang->setText($optionText);
            $valueLang->setProductFieldValue($productFieldValue);
            if (!empty($data['id_product_field_value'])) {
                $productFieldValue->getLangs()->add($valueLang);
            } else {
                $productFieldValue->setLangs(new ArrayCollection([$valueLang]));
            }
            $this->entityManager->persist($valueLang);
        }

        $this->entityManager->persist($productFieldValue);
        $this->entityManager->flush();

        return $productFieldValue;
        /*return [
            'idProduct' => $productFieldValue->getProductField()->getIdProduct(),
            'idProductField' => $productFieldValue->getProductField()->getId(),
        ];*/
    }

    public function delete(): PpbsProductFieldValue
    {
        $productFieldValue = $this->productFieldValueRepository->find($this->id);

        if (empty($productFieldValue)) {
            return [];
        }

        $this->deleteProductFieldValueLang($productFieldValue);
        $this->entityManager->remove($productFieldValue);
        $this->entityManager->flush();

        return $productFieldValue;
    }

    public function updatePositions(array $positions): void
    {
        foreach ($positions as $position) {
            $productFieldValue = $this->productFieldValueRepository->find((int) $position['rowId']);
            if (!empty($productFieldValue)) {
                $productFieldValue->setPosition((int) $position['newPosition']);
                $this->entityManager->persist($productFieldValue);
            }
        }
        $this->entityManager->flush();
    }
}
