<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    Musaffar Patel
 * @copyright 2016-2025 Musaffar Patel
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 */

declare(strict_types=1);

namespace PrestaShop\Module\ProductPriceBySize\Grid\Definition\Factory;

if (!defined('_PS_VERSION_')) {
    exit;
}

use PrestaShop\Module\ProductPriceBySize\Repository\ProductFieldRepository;
use PrestaShop\Module\ProductPriceBySize\Repository\ProductRepository;
use PrestaShop\PrestaShop\Core\Grid\Action\GridActionCollection;
use PrestaShop\PrestaShop\Core\Grid\Action\Row\RowActionCollection;
use PrestaShop\PrestaShop\Core\Grid\Action\Row\Type\LinkRowAction;
use PrestaShop\PrestaShop\Core\Grid\Action\Row\Type\SubmitRowAction;
use PrestaShop\PrestaShop\Core\Grid\Action\Type\LinkGridAction;
use PrestaShop\PrestaShop\Core\Grid\Column\ColumnCollection;
use PrestaShop\PrestaShop\Core\Grid\Column\Type\Common\ActionColumn;
use PrestaShop\PrestaShop\Core\Grid\Column\Type\Common\PositionColumn;
use PrestaShop\PrestaShop\Core\Grid\Column\Type\DataColumn;
use PrestaShop\PrestaShop\Core\Grid\Definition\Factory\AbstractGridDefinitionFactory;
use PrestaShop\PrestaShop\Core\Hook\HookDispatcherInterface;
use Symfony\Component\HttpFoundation\RequestStack;

class ProductFieldValueGridDefinitionFactory extends AbstractGridDefinitionFactory
{
    const GRID_ID = 'ppbs_product_field_value_x';

    private $idProduct = 0;

    private $IdProductField = 0;

    private $productRepository;

    private $productFieldRepository;

    private $product;

    public function __construct(
        HookDispatcherInterface $hookDispatcher,
        RequestStack $requestStack,
        ProductRepository $productRepository,
        ProductFieldRepository $productFieldRepository) {
        parent::__construct($hookDispatcher);
        $this->setProductFieldId($requestStack);
        $this->productRepository = $productRepository;
        $this->productFieldRepository = $productFieldRepository;
        $request = $requestStack->getCurrentRequest();

        if ((int) $request->get('id_product_field') > 0) {
            $this->product = $this->productFieldRepository->find((int) $request->get('id_product_field'))->getProduct();
        } elseif ((int) $request->get('id_product') > 0) {
            $this->product = $this->productRepository->find((int) $request->get('id_product'));
        }
    }

    /**
     * {@inheritdoc}
     */
    protected function getId()
    {
        return self::GRID_ID;
    }

    /**
     * {@inheritdoc}
     */
    protected function getName()
    {
        return $this->trans('Field Values', [], 'Modules.Productpricebysize.Admin');
    }

    /**
     * {@inheritdoc}
     */
    protected function getColumns()
    {
        return (new ColumnCollection())
            ->add(
                (new PositionColumn('position'))
                    ->setName($this->trans('Position', [], 'Admin.Global'))
                    ->setOptions([
                        'id_field' => 'id_product_field_value',
                        'position_field' => 'position',
                        'update_route' => 'productpricebysize_admin_product_ppbs_product_field_value_position',
                        'update_method' => 'POST',
                        'sortable' => false,
                        'record_route_params' => [
                            'id_product_field' => 'id_product_field',
                        ],
                    ])
            )
            ->add(
                (new DataColumn('id_product_field_value'))
                    ->setName($this->trans('Id', [], 'Modules.Productpricebysize.Admin'))
                    ->setOptions([
                        'field' => 'id_product_field_value',
                    ])
            )
            ->add(
                (new DataColumn('text'))
                    ->setName($this->trans('Text', [], 'Modules.Productpricebysize.Admin'))
                    ->setOptions([
                        'field' => 'text',
                    ])
            )
            ->add(
                (new DataColumn('value'))
                    ->setName($this->trans('Value', [], 'Modules.Productpricebysize.Admin'))
                    ->setOptions([
                        'field' => 'value',
                    ])
            )
            ->add(
                (new ActionColumn('actions'))
                    ->setOptions([
                        'actions' => (new RowActionCollection())
                            ->add(
                                (new LinkRowAction('edit'))
                                    ->setName($this->trans('Edit', [], 'Admin.Actions'))
                                    ->setIcon('edit')
                                    ->setOptions([
                                        'route' => 'productpricebysize_admin_product_ppbs_product_field_value_edit',
                                        'route_param_name' => 'id_product_field_value',
                                        'route_param_field' => 'id_product_field_value',
                                    ])
                            )
                        ->add(
                            (new SubmitRowAction('delete'))
                                ->setName($this->trans('Delete', [], 'Admin.Actions'))
                                ->setIcon('delete')
                                ->setOptions([
                                    'method' => 'POST',
                                    'route' => 'productpricebysize_admin_product_common_product_field_value_delete',
                                    'route_param_name' => 'id_product_field_value',
                                    'route_param_field' => 'id_product_field_value',
                                    'confirm_message' => $this->trans('Remove this field value?', [], 'Modules.Productpricebysize.Admin'),
                                ])
                        ),
                    ])
            );
    }

    protected function getGridActions()
    {
        switch ($this->product->getType()) {
            case 'pap':
                $route = 'productpricebysize_admin_product_pap_product_field_value_new';
                break;
            default:
                $route = 'productpricebysize_admin_product_ppbs_product_field_value_new';
                break;
        }

        return (new GridActionCollection())
            ->add(
                (new LinkGridAction('value_new'))
                    ->setName($this->trans('Add New Value', [], 'Modules.Productbundlespro.Admin'))
                    ->setIcon('add_circle')
                    ->setOptions([
                        'route' => $route,
                        'route_params' => [
                            'id_product_field' => $this->IdProductField,
                        ],
                    ])
            );
    }

    private function setProductFieldId(RequestStack $requestStack)
    {
        $request = $requestStack->getCurrentRequest();
        if (null !== $request && (int) $request->get('id_product_field') > 0) {
            $this->IdProductField = $request->get('id_product_field');
        }
    }
}
