<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    Musaffar Patel
 * @copyright 2016-2025 Musaffar Patel
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 */

declare(strict_types=1);

namespace PrestaShop\Module\ProductPriceBySize\Presenter\Front\Common;

if (!defined('_PS_VERSION_')) {
    exit;
}

class AbstractProductPresenter
{
    protected $module;

    protected $context;

    protected $productRepository;

    protected $productPackRepository;

    protected $product;

    protected $productPack;

    protected $productFieldsRepository;

    protected $unitRepository;

    protected $equationRepository;

    protected $equationVariableRepository;

    protected $productUnits = [];

    public function __construct(\Module $module, \Context $context)
    {
        $this->module = $module;
        $this->context = $context;
        $idProduct = (int) \Tools::getValue('id_product');

        $this->productRepository = $this->module->get('prestashop.module.productpricebysize.repository.product_repository');
        if ($idProduct) {
            $this->product = $this->productRepository->find($idProduct);
        }

        $this->productPackRepository = $this->module->get('prestashop.module.productpricebysize.repository.product_pack_repository');
        $this->productFieldsRepository = $this->module->get('prestashop.module.productpricebysize.repository.product_field_repository');
        $this->unitRepository = $this->module->get('prestashop.module.productpricebysize.repository.unit_repository');
        $this->equationRepository = $this->module->get('prestashop.module.productpricebysize.repository.equation_repository');
        $this->equationVariableRepository = $this->module->get('prestashop.module.productpricebysize.repository.equation_variable_repository');

        $this->productPack = null;
        foreach ($this->product->getProductPacks() as $productPack) {
            if ($productPack->getIdProductAttribute() == 0) {
                $this->productPack = $productPack;
                break;
            }
        }
        // $this->productPack = $this->productPackRepository->get($idProduct, 0);

        $productUnits = $this->product->getProductUnits();
        foreach ($productUnits as $productUnit) {
            $unit = $this->unitRepository->findByLang($productUnit->getUnit()->getId(), $this->context->language->id);
            $this->productUnits[] = [
                'id' => $productUnit->getId(),
                'idUnit' => $unit['id'],
                'name' => $unit['name'],
                'conversionFactor' => $unit['conversionFactor'],
                'lang' => $unit['langs'][0],
                'isDefault' => $productUnit->getIsDefault(),
            ];
        }
    }

    public function getProductInfo(int $idProduct, int $idProductAttribute, int $quantity): array
    {
        $stockRepository = $this->module->get('prestashop.module.productpricebysize.repository.stock_repository');

        // customer group reduction
        $customerGroupReduction = 0;
        if ((int) $this->context->customer->id > 0) {
            $customerGroupReduction = \Group::getReduction($this->context->customer->id);
        }

        $ppbsStock = $stockRepository->get($idProduct, $idProductAttribute, $this->context->shop->id);

        $product_obj = new \Product($idProduct);
        $product = [];
        $product['id_product'] = $idProduct;
        $product['id_product_attribute'] = $idProductAttribute;
        $product['out_of_stock'] = $product_obj->out_of_stock;
        $product['id_category_default'] = $product_obj->id_category_default;
        $product['link_rewrite'] = ''; // $product_obj->link_rewrite;
        $product['ean13'] = $product_obj->ean13;
        $product['minimal_quantity'] = $product_obj->minimal_quantity;
        $product['unit_price_ratio'] = $product_obj->unit_price_ratio;
        $product['price_display'] = (int) \Product::getTaxCalculationMethod($this->context->customer->id);
        $product['quantity_wanted'] = $quantity;
        $product['group_reduction'] = $customerGroupReduction;
        $product['ppbs_stock'] = $ppbsStock;

        $product_properties = \Product::getProductProperties($this->context->language->id, $product, null);
        $product['id_product_attribute'] = 0;
        $base_price = $product_obj->price;

        if ($customerGroupReduction > 0) {
            $base_price = $base_price / ((100 - $customerGroupReduction) / 100);
        }

        $product_properties['base_price_exc_tax'] = $base_price;
        // it's possible some specific prices (Fixed impact) need to override the main attribute price
        if (!empty($product_properties['specific_prices']['id_specific_price'])) {
            $specific_price = $product_properties['specific_prices'];
            if ($specific_price['reduction'] == 0 && $specific_price['price'] > 0) {
                $product_properties['attribute_price'] = 0;
            }
        }

        return $product_properties;
    }

    public function hookDisplayHeader(array $params, int $idProduct): void
    {
        $this->context->controller->addCSS($this->module->getPathUri() . 'views/css/front/global.css');
        $this->context->controller->addCSS($this->module->getPathUri() . 'views/css/front/ppbs.css');
        $this->context->controller->addCSS($this->module->getPathUri() . 'views/css/front/pap.css');
        $this->context->controller->addJquery();
        $this->context->controller->addJS($this->module->getPathUri() . 'views/js/front/common/MpTools.js');
        $this->context->controller->addJS($this->module->getPathUri() . 'views/js/front/common/FrontProductController.js');
        $this->context->controller->addJS($this->module->getPathUri() . 'views/js/front/ppbs/PpbsFrontProductController.js');
    }
}
