<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    Musaffar Patel
 * @copyright 2016-2025 Musaffar Patel
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 */

declare(strict_types=1);

namespace PrestaShop\Module\ProductPriceBySize\Presenter\Front\Pap;

use PrestaShop\Module\ProductPriceBySize\Presenter\Front\Common\AbstractProductPresenter;
use PrestaShop\Module\ProductPriceBySize\Service\ArrayService;
use PrestaShop\Module\ProductPriceBySize\Service\AssetsService;

if (!defined('_PS_VERSION_')) {
    exit;
}

class ProductPresenter extends AbstractProductPresenter
{
    public function present(int $idProduct): array
    {
        if (empty($this->productPack)) {
            return [];
        }
        $productFields = $this->productFieldsRepository->getByProduct($idProduct, $this->context->language->id);

        $units = $this->unitRepository->findAllByLang($this->context->language->id);
        $unit = $this->unitRepository->findByLang($this->product->getUnit()->getId(), $this->context->language->id);
        $wastageEntities = $this->product->getWastageOptions();
        $wastageOptions = [];

        foreach ($wastageEntities as $wastageEntity) {
            $wastageOptions[] = [
                'id' => $wastageEntity->getWastage()->getId(),
                'value' => $wastageEntity->getWastage()->getValue(),
            ];
        }

        $totalAreaSuffix = '';
        if (count($productFields) > 1) {
            $totalAreaSuffix = count($productFields);
        }

        return [
            'productFields' => $productFields,
            'productUnits' => $this->productUnits,
            'idProduct' => $idProduct,
            'idShop' => $this->context->shop->id,
            'productInfo' => json_encode([]),
            'unit' => $unit,
            'units' => json_encode(ArrayService::indexBy('name', $units)),
            'productPack' => json_encode($this->productPack),
            'wastageOptions' => $wastageOptions,
            'totalAreaSuffix' => $totalAreaSuffix,
            'imagesUrl' => AssetsService::getImagesUrl(),
        ];
    }

    public function renderHookDisplayProductAdditionalInfo(&$smartyInstance, array $presented): string
    {
        if (empty($this->productPack)) {
            return '';
        }

        $smartyInstance->assign($presented);
        switch ($this->productPack->getProduct()->getPapType()) {
            case 'normal':
                return $this->module->display($this->module->moduleFile, 'views/templates/front/pap/product/normal.tpl');
            case 'rolls':
                return $this->module->display($this->module->moduleFile, 'views/templates/front/pap/product/rolls.tpl');
            case 'paint':
                return $this->module->display($this->module->moduleFile, 'views/templates/front/pap/product/paint.tpl');
        }

        return '';
    }

    public function getPackInfo(int $idProduct, int $idProductAttribute): array
    {
        $product = $this->productRepository->find($idProduct);
        $productPack = $this->productPackRepository->get($idProduct, $idProductAttribute, true);

        if (empty($productPack)) {
            return [];
        }

        $packInfo = [
            'idProduct' => $idProduct,
            'idProductAttribute' => $idProductAttribute,
            'type' => $product->getPapType(),
            'packArea' => $productPack['packArea'],
            'areaPrice' => $productPack['areaPrice'],
            'rollHeight' => $productPack['rollHeight'],
            'rollWidth' => $productPack['rollWidth'],
            'patternRepeat' => $productPack['patternRepeat'],
            'coverage' => $productPack['coverage'],
            'isDynamicPrice' => $product->getIsDynamicPrice(),
        ];

        return $packInfo;
    }

    public function hookDisplayHeader(array $params, int $idProduct): void
    {
        if ($this->context->controller->php_self != 'product' && $this->context->controller->php_self != 'category' && $this->context->controller->php_self != 'search') {
            return;
        }
        parent::hookDisplayHeader($params, $idProduct);
        $this->context->controller->addJS($this->module->getPathUri() . 'views/js/front/ppbs/PpbsFrontProductController.js');
        $this->context->controller->addJS($this->module->getPathUri() . 'views/js/front/pap/PapFrontProductCoreController.js');
        $this->context->controller->addJS($this->module->getPathUri() . 'views/js/front/pap/PapFrontProductNormalController.js');
        $this->context->controller->addJS($this->module->getPathUri() . 'views/js/front/pap/PapFrontProductRollsController.js');
        $this->context->controller->addJS($this->module->getPathUri() . 'views/js/front/pap/PapFrontProductPaintsController.js');
    }
}
