<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    Musaffar Patel
 * @copyright 2016-2025 Musaffar Patel
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 */

declare(strict_types=1);

namespace PrestaShop\Module\ProductPriceBySize\Presenter\Front\Ppbs;

use PrestaShop\Module\ProductPriceBySize\Enum\EquationType;
use PrestaShop\Module\ProductPriceBySize\Presenter\Front\Common\AbstractProductPresenter;
use PrestaShop\Module\ProductPriceBySize\Presenter\Front\Common\ProductPresenterInterface;
use PrestaShop\Module\ProductPriceBySize\Service\ArrayService;
use PrestaShop\Module\ProductPriceBySize\Service\AssetsService;

if (!defined('_PS_VERSION_')) {
    exit;
}

class ProductPresenter extends AbstractProductPresenter implements ProductPresenterInterface
{
    public function present(int $idProduct): array
    {
        // $entityManager = $this->module->get('doctrine.orm.entity_manager');
        $productFields = $this->productFieldsRepository->getByProduct($idProduct, $this->context->language->id);
        $units = $this->unitRepository->findAllByLang($this->context->language->id);
        $unit = $this->unitRepository->findByLang($this->product->getUnit()->getId(), $this->context->language->id);
        $equations = $this->equationRepository->getAllByProduct($idProduct, EquationType::PRICE);
        $equationVariables = $this->equationVariableRepository->findAll();
        $areaPrices = [];
        $equationsIndexed = [];
        $equationVariablesIndexed = [];

        $this->productRepository->setCustomizationFieldModuleFlag($idProduct);

        foreach ($equationVariables as $equationVariable) {
            $equationVariablesIndexed[] = [
                'id' => $equationVariable->getId(),
                'name' => $equationVariable->getName(),
                'value' => $equationVariable->getValue(),
            ];
        }

        foreach ($equations as $equation) {
            if ($equation['idProductAttribute'] > 0) {
                $equationsIndexed[$equation['idProductAttribute']] = $equation;
            } else {
                $equationsIndexed[0] = $equation;
            }
        }

        foreach ($this->product->getAreaPrices() as $areaPrice) {
            $areaPrices[] = [
                'id' => $areaPrice->getId(),
                'idShop' => $areaPrice->getIdShop(),
                'idProduct' => $areaPrice->getIdProduct(),
                'areaLow' => $areaPrice->getAreaLow(),
                'areaHigh' => $areaPrice->getAreaHigh(),
                'impact' => $areaPrice->getImpact(),
                'price' => $areaPrice->getPrice(),
                'weight' => $areaPrice->getWeight(),
            ];
        }

        foreach ($productFields as &$productField) {
            $productField['defaultValue'] = round((float)$productField['defaultValue'], (int)$productField['decimals']);;
        }

        $presented = [
            'idProduct' => $idProduct,
            'idShop' => $this->context->shop->id,
            'minTotalArea' => $this->product->getMinTotalArea(),
            'minPrice' => $this->product->getMinPrice(),
            'setupFee' => $this->product->getSetupFee(),
            'isAttributePriceAreaPrice' => (int) $this->product->getIsAttributePriceAreaPrice(),
            'isStockEnabled' => (int) $this->product->getIsStockEnabled(),
            'productFields' => $productFields,
            'productInfo' => json_encode([]),
            'unit' => json_encode($unit),
            'units' => json_encode(ArrayService::indexBy('name', $units)),
            'productUnits' => $this->productUnits,
            'areaPrices' => json_encode($areaPrices),
            'equations' => json_encode($equationsIndexed),
            'equationVariables' => json_encode($equationVariablesIndexed),
            'imagesUrl' => AssetsService::getImagesUrl(),
        ];

        return $presented;
    }

    public function renderHookDisplayProductAdditionalInfo(&$smartyInstance, array $presented): string
    {
        $smartyInstance->assign($presented);

        return $this->module->display($this->module->moduleFile, 'views/templates/front/ppbs/product/widget.tpl');
    }

    public function hookDisplayHeader(array $params, int $idProduct): void
    {
        if ($this->context->controller->php_self != 'product' && $this->context->controller->php_self != 'category' && $this->context->controller->php_self != 'search') {
            return;
        }
        parent::hookDisplayHeader($params, $idProduct);
        $this->context->controller->addJS($this->module->getPathUri() . 'views/js/front/ppbs/PpbsFrontProductController.js');
    }
}
