<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    Musaffar Patel
 * @copyright 2016-2025 Musaffar Patel
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 */

declare(strict_types=1);

namespace PrestaShop\Module\ProductPriceBySize\Repository;

use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use PrestaShop\Module\ProductPriceBySize\Service\FormatterService;

if (!defined('_PS_VERSION_')) {
    exit;
}

class CartRepository extends ServiceEntityRepository
{
    private $connection;

    private $databasePrefix;

    private $languages = [];

    private $translator;

    public function __construct($registry, $connection, $databasePrefix, $translator, array $languages)
    {
        // parent::__construct($registry, ProductCommentCriterion::class);
        // $container = (new ContainerFinder(\Context::getContext()))->getContainer();
        $this->connection = $connection;
        $this->databasePrefix = $databasePrefix;
        $this->languages = $languages;
        $this->translator = $translator;
    }

    public function getCustomizationDisplayText(array $cartData): string
    {
        $value = '';
        if (!empty($cartData)) {
            foreach ($cartData['fields'] as $productField) {
                $value .= $productField['dimension']['displayName'] . ' : ' . $productField['value'] . ' ' . $productField['unit']['displayName'] . '. ';
            }
        }

        return $value;
    }

    public function getCustomizationField(int $idProduct, int $idShop): int
    {
        $queryBuilder = $this->connection->createQueryBuilder();
        $queryBuilder
            ->select('cf.*')
            ->from($this->databasePrefix . 'customization_field', 'cf')
            ->innerJoin('cf', $this->databasePrefix . 'customization_field_lang', 'cfl', 'cfl.id_customization_field = cf.id_customization_field')
            ->where('cf.id_product = :idProduct')
            ->andWhere('cf.is_ppbs = 1')
            ->groupBy('cf.id_customization_field')
            ->setParameter('idProduct', $idProduct);
        // $row = $queryBuilder->execute()->fetchAssociative();  PS 8 only
        $row = $queryBuilder->execute()->fetch();

        if (empty($row)) {
            $this->connection->insert($this->databasePrefix . 'customization_field', [
                'id_product' => $idProduct,
                'type' => 1,
                'required' => 0,
                'is_module' => 1,
                'is_deleted' => 0,
                'is_ppbs' => 1,
            ]);
            $idCustomizationField = $this->connection->lastInsertId();

            foreach ($this->languages as $language) {
                $this->connection->insert($this->databasePrefix . 'customization_field_lang', [
                    'id_customization_field' => (int) $idCustomizationField,
                    'id_lang' => (int) $language['id_lang'],
                    'id_shop' => $idShop,
                    'name' => $this->translator->trans('options', [], 'Module.Productpricebysize.Front'),
                ]);
            }
        } else {
            $idCustomizationField = $row['id_customization_field'];
            foreach ($this->languages as $language) {
                $this->connection->update(
                    $this->databasePrefix . 'customization_field_lang',
                    ['name' => pSQL($this->translator->trans('options', [], 'Module.Productpricebysize.Front'))],
                    ['id_customization_field' => $idCustomizationField]
                );
            }
        }

        return (int) $idCustomizationField;
    }

    public function addCustomizedData(int $idCustomization, int $idCustomizationField, string $customizedDataText, int $idModule, array $cartData)
    {
        $this->connection->insert($this->databasePrefix . 'customized_data', [
            'id_customization' => $idCustomization,
            'type' => 1,
            '`index`' => $idCustomizationField,
            'value' => pSQL($customizedDataText),
            'ppbs_data' => pSQL(FormatterService::rawJsonEncode($cartData), true),
            'id_module' => $idModule,
        ]);
    }

    public function addCustomization(int $idProduct, int $idCart, int $ipa, int $idAddressDelivery, array $cartData, int $idModule, int $quantity, int $idShop = 1): int
    {
        if (empty($cartData) || $idProduct == 0 || $idCart == 0) {
            return 0;
        }

        if (version_compare(_PS_VERSION_, '1.7.8.12', '<')) {
            $quantity = 0;
        }

        $this->connection->insert($this->databasePrefix . 'customization', [
            'id_product_attribute' => $ipa,
            'id_address_delivery' => $idAddressDelivery,
            'id_cart' => $idCart,
            'id_product' => $idProduct,
            'quantity' => $quantity,
            'in_cart' => 1,
        ]);
        $idCustomization = (int) $this->connection->lastInsertId();
        $value = $this->getCustomizationDisplayText($cartData);
        $idCustomizationField = self::getCustomizationField($idProduct, $idShop);
        $this->addCustomizedData($idCustomization, $idCustomizationField, $value, $idModule, $cartData);

        $this->connection->update(
            $this->databasePrefix . 'cart_product',
            ['quantity' => 1],
            [
                'id_cart' => $idCart,
                'id_product' => $idProduct,
                'id_product_attribute' => $ipa,
                'id_shop' => $idShop,
                'id_customization' => $idCustomization,
            ]
        );

        return $idCustomization;
    }

    public function getCartData(int $idProduct, int $idCart, int $idProductAttribute, int $idShop, int $idCustomization = 0): array
    {
        $cartData = [];
        if (empty($idProduct) || empty($idCart)) {
            return [];
        }

        $queryBuilder = $this->connection->createQueryBuilder();
        $queryBuilder
            ->select('cd.ppbs_data', 'c.quantity')
            ->from($this->connection->getDatabasePlatform()->quoteSingleIdentifier($this->databasePrefix . 'customized_data'), 'cd')
            ->innerJoin(
                'cd',
                $this->connection->getDatabasePlatform()->quoteSingleIdentifier($this->databasePrefix . 'customization'),
                'c',
                'cd.id_customization = c.id_customization'
            )
            ->where('c.id_cart = :id_cart')
            ->andWhere('c.id_product = :id_product')
            ->andWhere('c.id_product_attribute = :id_product_attribute')
            ->andWhere('cd.ppbs_data <> ""')
            ->setParameter(':id_cart', $idCart)
            ->setParameter(':id_product', $idProduct)
            ->setParameter(':id_product_attribute', $idProductAttribute);

        if ($idCustomization > 0) {
            $queryBuilder->andWhere('cd.id_customization = :id_customization')
                ->setParameter(':id_customization', $idCustomization);
        }
        // $rows = $queryBuilder->execute()->fetchAllAssociative(); PS 8 / DBAL 3
        $rows = $queryBuilder->execute()->fetchAll();
        if (!empty($rows)) {
            foreach ($rows as $row) {
                $cartData[] = [
                    'productFields' => json_decode(stripslashes($row['ppbs_data']), true),
                    'quantity' => $row['quantity'],
                ];
            }
        }

        return $cartData;
    }

    public function getProductQuantity(int $idProduct, int $idProductAttribute, int $idCustomization, int $idCart): int
    {
        $queryBuilder = $this->connection->createQueryBuilder();
        $queryBuilder
            ->select('quantity as total_quantity')
            ->from($this->connection->getDatabasePlatform()->quoteSingleIdentifier($this->databasePrefix . 'cart_product'))
            ->where('id_cart = :idCart')
            ->andWhere('id_product = :idProduct')
            ->andWhere('id_product_attribute = :idProductAttribute')
            ->andWhere('id_customization = :idCustomization')
            ->setParameter(':idCart', $idCart)
            ->setParameter(':idProduct', $idProduct)
            ->setParameter(':idProductAttribute', $idProductAttribute)
            ->setParameter(':idCustomization', $idCustomization);
        // $row = $queryBuilder->execute()->fetchAssociative();
        $row = $queryBuilder->execute()->fetch();

        return (int) $row['total_quantity'];
    }

    public function setProductCustomizationQuantity(int $idProduct, int $idProductAttribute, int $idCustomization, int $idCart, int $quantity): bool
    {
        $this->connection->update(
            $this->databasePrefix . 'customization',
            ['quantity' => $quantity],
            [
                'id_cart' => $idCart,
                'id_product' => $idProduct,
                'id_product_attribute' => $idProductAttribute,
                'id_customization' => $idCustomization,
            ]
        );

        return true;
    }
}
