<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    Musaffar Patel
 * @copyright 2016-2025 Musaffar Patel
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 */

declare(strict_types=1);

namespace PrestaShop\Module\ProductPriceBySize\Repository;

use Doctrine\ORM\EntityRepository;

if (!defined('_PS_VERSION_')) {
    exit;
}

class ProductFieldRepository extends EntityRepository
{
    private $registry;

    private $connection;

    private $databasePrefix;

    public function getOneByLang(int $idProductField, int $idLang): array
    {
        $entityManager = $this->getEntityManager();
        $query = $entityManager->createQuery(
            'SELECT pf, d, dl, ft, u, ul
            FROM PrestaShop\Module\ProductPriceBySize\Entity\PpbsProductField pf
            INNER JOIN pf.dimension d
            INNER JOIN d.dimensionLangs dl            
            INNER JOIN pf.fieldType ft            
            INNER JOIN pf.unit u            
            INNER JOIN u.langs ul            
            WHERE pf.id = :idProductField
            AND dl.idLang = :idLang AND ul.idLang = :idLang'
        )
            ->setParameter('idLang', $idLang)
            ->setParameter('idProductField', $idProductField);
        $query->setMaxResults(1);
        try {
            $result = $query->getSingleResult(\Doctrine\ORM\AbstractQuery::HYDRATE_ARRAY);
            if (!empty($result['dimension']['dimensionLangs'][0])) {
                $result['dimension']['lang'] = $result['dimension']['dimensionLangs'][0];
            }
            if (!empty($result['unit']['langs'][0])) {
                $result['unit']['lang'] = $result['unit']['langs'][0];
            }
        } catch (\Doctrine\ORM\NoResultException $e) {
            $result = [];
        }

        return $result;
    }

    public function getByProduct(int $idProduct, int $idLang): array
    {
        $entityManager = $this->getEntityManager();
        $query = $entityManager->createQuery(
            'SELECT pf, d, dl, ft, u, ul, pfv, pfvl
            FROM PrestaShop\Module\ProductPriceBySize\Entity\PpbsProductField pf
            INNER JOIN pf.dimension d
            INNER JOIN d.dimensionLangs dl            
            INNER JOIN pf.fieldType ft            
            INNER JOIN pf.unit u            
            INNER JOIN u.langs ul
            LEFT JOIN pf.values pfv            
            LEFT JOIN pfv.langs pfvl WITH pfvl.idLang = :idLang            
            WHERE pf.idProduct = :idProduct
            AND dl.idLang = :idLang AND ul.idLang = :idLang
            ORDER BY pf.position'
        )->setParameter('idProduct', $idProduct)
        ->setParameter('idLang', $idLang);

        $result = $query->getArrayResult();

        foreach ($result as &$field) {
            if (!empty($field['dimension']['dimensionLangs'][0])) {
                $field['dimension']['lang'] = $field['dimension']['dimensionLangs'][0];
                unset($field['dimension']['dimensionLangs']);
            }
            if (!empty($field['unit']['langs'][0])) {
                $field['unit']['lang'] = $field['unit']['langs'][0];
            }

            if ($field['fieldType']['name'] == 'dropdown') {
                foreach ($field['values'] as &$value) {
                    if (!empty($value['langs'][0])) {
                        $value['lang'] = $value['langs'][0];
                    }
                }
            }
        }

        return $result;
    }

    public function getNewPosition(int $idProduct): int
    {
        $entityManager = $this->getEntityManager();
        $query = $entityManager->createQuery(
            'SELECT MAX(pf.position) as maxPosition
            FROM PrestaShop\Module\ProductPriceBySize\Entity\PpbsProductField pf
            WHERE pf.idProduct = :idProduct'
        )->setParameter('idProduct', $idProduct);

        $result = $query->getArrayResult();

        return (int) $result[0]['maxPosition'];
    }
}
