<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    Musaffar Patel
 * @copyright 2016-2025 Musaffar Patel
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 */

declare(strict_types=1);

namespace PrestaShop\Module\ProductPriceBySize\Repository;

use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\EntityRepository;
use Doctrine\ORM\Mapping\ClassMetadata;
use PrestaShop\PrestaShop\Adapter\ContainerFinder;

if (!defined('_PS_VERSION_')) {
    exit;
}

class ProductRepository extends EntityRepository
{
    private $connection;

    private $databasePrefix;

    public function __construct(EntityManagerInterface $em, ClassMetadata $class)
    {
        parent::__construct($em, $class);
        $container = (new ContainerFinder(\Context::getContext()))->getContainer();
        if ($container !== null) {
            $this->connection = $container->get('doctrine.dbal.default_connection');
        }
        $this->databasePrefix = $container->getParameter('database_prefix');
    }

    public function getAttributePriceData(int $idProduct, int $idShop): array
    {
        $queryBuilder = $this->connection->createQueryBuilder();
        $queryBuilder
            ->select($this->databasePrefix . 'product_shop.price', $this->databasePrefix . 'product_shop.ecotax')
            ->from($this->databasePrefix . 'product', 'p')
            ->innerJoin('p', $this->databasePrefix . 'product_shop', $this->databasePrefix . 'product_shop', $this->databasePrefix . 'product_shop.id_product = p.id_product AND ' . $this->databasePrefix . 'product_shop.id_shop = :id_shop')
            ->where('p.id_product = :id_product')
            ->setParameter('id_shop', $idShop)
            ->setParameter('id_product', $idProduct);

        if (\Combination::isFeatureActive()) {
            $queryBuilder
                ->addSelect($this->databasePrefix . 'product_attribute_shop.id_product_attribute', $this->databasePrefix . 'product_attribute_shop.price AS attribute_price', $this->databasePrefix . 'product_attribute_shop.default_on')
                ->leftJoin('p', $this->databasePrefix . 'product_attribute', 'pa', 'pa.id_product = p.id_product')
                ->leftJoin('pa', $this->databasePrefix . 'product_attribute_shop', $this->databasePrefix . 'product_attribute_shop', $this->databasePrefix . 'product_attribute_shop.id_product_attribute = pa.id_product_attribute AND ' . $this->databasePrefix . 'product_attribute_shop.id_shop = :id_shop');
        } else {
            $queryBuilder->addSelect('0 as id_product_attribute');
        }
        // $result = $queryBuilder->execute()->fetchAllAssociative();
        $result = $queryBuilder->execute()->fetchAll();

        return !empty($result) ? $result : [];
    }

    public function deleteProductData(int $idProduct)
    {
        $queryBuilder = $this->connection->createQueryBuilder();
        $queryBuilder
            ->delete($this->databasePrefix . 'ppbs_product')
            ->where('id_product = :idProduct')
            ->setParameter('idProduct', $idProduct);
        $queryBuilder->execute();
    }

    public function deleteProductPackData(int $idProduct)
    {
        $queryBuilder = $this->connection->createQueryBuilder();
        $queryBuilder
            ->delete($this->databasePrefix . 'ppbs_product_pack')
            ->where('id_product = :idProduct')
            ->setParameter('idProduct', $idProduct);
        $queryBuilder->execute();
    }

    public function deleteProductUnits(int $idProduct)
    {
        $queryBuilder = $this->connection->createQueryBuilder();
        $queryBuilder
            ->delete($this->databasePrefix . 'ppbs_product_unit')
            ->where('id_product = :id_product')
            ->setParameter('id_product', $idProduct);
        $queryBuilder->execute();
    }

    public function deleteProductWastage(int $idProduct)
    {
        $queryBuilder = $this->connection->createQueryBuilder();
        $queryBuilder
            ->delete($this->databasePrefix . 'ppbs_product_wastage')
            ->where('id_product = :id_product')
            ->setParameter('id_product', $idProduct);
        $queryBuilder->execute();
    }

    public function deleteProductEquations(int $idProduct): void
    {
        $query = $this->getEntityManager()->createQuery(
            'DELETE FROM PrestaShop\Module\ProductPriceBySize\Entity\PpbsEquationTemplate et
             WHERE et.isGlobal = 0
             AND et.id IN (
                SELECT et.id
                FROM PrestaShop\Module\ProductPriceBySize\Entity\PpbsProductEquation pe
                WHERE pe.idEquationTemplate = et.id
                AND pe.idProduct = :idProduct
             )'
        )->setParameter('idProduct', $idProduct);
        $query->execute();

        $query = $this->getEntityManager()->createQuery(
            'DELETE FROM PrestaShop\Module\ProductPriceBySize\Entity\PpbsProductEquation pe
             WHERE pe.idProduct = :idProduct'
        )->setParameter('idProduct', $idProduct);
        $query->execute();
    }

    public function deleteProductAreaPrices(int $idProduct): void
    {
        $queryBuilder = $this->connection->createQueryBuilder();
        $queryBuilder
            ->delete($this->databasePrefix . 'ppbs_area_price')
            ->where('id_product = :id_product')
            ->setParameter('id_product', $idProduct);
        $queryBuilder->execute();
    }

    public function getProductsByCategory(int $idCategory, int $idLang): array
    {
        $queryBuilder = $this->connection->createQueryBuilder();
        $queryBuilder
            ->select('p.id_product, pl.name, p.reference')
            ->from($this->databasePrefix . 'category_product', 'cp')
            ->innerJoin('cp', $this->databasePrefix . 'product', 'p', 'p.id_product = cp.id_product')
            ->innerJoin('cp', $this->databasePrefix . 'product_lang', 'pl', 'pl.id_product = cp.id_product')
            ->where('cp.id_category = :id_category')
            ->andWhere('pl.id_lang = :id_lang')
            ->setParameter('id_category', $idCategory)
            ->setParameter('id_lang', $idLang);
        // $result = $queryBuilder->execute()->fetchAllAssociative();
        $result = $queryBuilder->execute()->fetchAll();

        return !empty($result) ? $result : [];
    }

    public function setCustomizationFieldModuleFlag(int $idProduct): void
    {
        $qb = $this->connection->createQueryBuilder();
        $qb->update($this->databasePrefix . 'customization_field', 'cf')
            ->set('cf.is_module', ':isModule')
            ->where('cf.id_product = :idProduct')
            ->andWhere('cf.is_ppbs = :isPpbs')
            ->setParameter('isModule', 1)
            ->setParameter('idProduct', $idProduct)
            ->setParameter('isPpbs', 1);
        $qb->execute();
    }
}
