<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    Musaffar Patel
 * @copyright 2016-2025 Musaffar Patel
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 */

declare(strict_types=1);

namespace PrestaShop\Module\ProductPriceBySize\Schema;

if (!defined('_PS_VERSION_')) {
    exit;
}

class Schema
{
    public $schema = [];

    private $data = [
        'dimensions' => [
            [
                'name' => 'width',
                'lang' => [
                    'text' => 'Width',
                ],
            ],
            [
                'name' => 'height',
                'lang' => [
                    'text' => 'Height',
                ],
            ],
            [
                'name' => 'depth',
                'lang' => [
                    'text' => 'Depth',
                ],
            ],
            [
                'name' => 'diameter',
                'lang' => [
                    'text' => 'Diameter',
                ],
            ],
        ],
        'field_type' => [
            'textbox',
            'dropdown',
        ],
        'units' => [
            [
                'name' => 'millimeter',
                'conversion_factor' => 1.00,
                'lang' => [
                    'text' => 'mm',
                ],
            ],
            [
                'name' => 'centimeter',
                'conversion_factor' => 10.00,
                'lang' => [
                    'text' => 'cm',
                ],
            ],
            [
                'name' => 'meter',
                'conversion_factor' => 1000.00,
                'lang' => [
                    'text' => 'm',
                ],
            ],
            [
                'name' => 'inch',
                'conversion_factor' => 25.40,
                'lang' => [
                    'text' => 'inch',
                ],
            ],
            [
                'name' => 'foot',
                'conversion_factor' => 304.80,
                'lang' => [
                    'text' => 'ft',
                ],
            ],
        ],
        'equation_type' => [
            [
                'name' => 'price',
            ],
            [
                'name' => 'weight',
            ],
        ],
    ];

    public function __construct()
    {
        $this->schema = [
            'ppbs_area_price' => [
                'ddl' => '
                    CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . "ppbs_area_price` (
                      `id_area_price` mediumint unsigned NOT NULL AUTO_INCREMENT,
                      `id_shop` smallint unsigned DEFAULT '0',
                      `id_product` mediumint unsigned NOT NULL DEFAULT '0',
                      `area_low` decimal(8,2) unsigned NOT NULL DEFAULT '0.00',
                      `area_high` decimal(8,2) unsigned NOT NULL DEFAULT '0.00',
                      `impact` varchar(6) COLLATE utf8mb4_general_ci DEFAULT NULL,
                      `price` decimal(8,2) unsigned NOT NULL DEFAULT '0.00',
                      `weight` decimal(8,2) unsigned NOT NULL DEFAULT '0.00',
                      PRIMARY KEY (`id_area_price`)                    
                    ) ENGINE=" . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8 ;            
                ', ],
            'ppbs_dimension' => [
                'ddl' => '
                    CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . "ppbs_dimension` (
                      `id_dimension` int unsigned NOT NULL AUTO_INCREMENT,
                      `name` varchar(64) NOT NULL,
                      `is_deleted` tinyint unsigned NOT NULL DEFAULT '0',
                      PRIMARY KEY (`id_dimension`)                    
                    ) ENGINE=" . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8 ;            
                ', ],
            'ppbs_dimension_lang' => [
                'ddl' => '
                    CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . 'ppbs_dimension_lang` (
                      `id_dimension` int unsigned NOT NULL,
                      `id_lang` int unsigned NOT NULL,
                      `display_name` varchar(128) COLLATE utf8mb4_general_ci NOT NULL,
                      `image` varchar(512) COLLATE utf8mb4_general_ci DEFAULT NULL,
                      PRIMARY KEY (`id_dimension`,`id_lang`)                    
                    ) ENGINE=' . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8 ;            
                ', ],
            'ppbs_equation_template' => [
                'ddl' => '
                    CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . "ppbs_equation_template` (
                      `id_equation_template` mediumint unsigned NOT NULL AUTO_INCREMENT,
                      `name` varchar(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
                      `equation` varchar(2056) COLLATE utf8mb4_general_ci DEFAULT NULL,
                      `id_equation_type` smallint unsigned NOT NULL DEFAULT '0',
                      `is_global` tinyint unsigned DEFAULT '0',
                      PRIMARY KEY (`id_equation_template`)                    
                    ) ENGINE=" . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8 ;            
                ', ],
            'ppbs_equation_type' => [
                'ddl' => '
                    CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . 'ppbs_equation_type` (
                      `id_equation_type` smallint unsigned NOT NULL AUTO_INCREMENT,
                      `name` varchar(16) COLLATE utf8mb4_general_ci NOT NULL,
                      PRIMARY KEY (`id_equation_type`)                    
                    ) ENGINE=' . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8 ;            
                ', ],
            'ppbs_equation_variable' => [
                'ddl' => '
                    CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . "ppbs_equation_variable` (
                      `id_equation_variable` smallint unsigned NOT NULL AUTO_INCREMENT,
                      `name` varchar(128) COLLATE utf8mb4_general_ci DEFAULT NULL,
                      `value` decimal(8,2) unsigned DEFAULT '0.00',
                      PRIMARY KEY (`id_equation_variable`)                    
                    ) ENGINE=" . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8 ;            
                ', ],
            'ppbs_field_type' => [
                'ddl' => '
                    CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . 'ppbs_field_type` (
                      `id_field_type` smallint unsigned NOT NULL AUTO_INCREMENT,
                      `name` varchar(32) CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci DEFAULT NULL,
                      PRIMARY KEY (`id_field_type`)                    
                    ) ENGINE=' . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8 ;            
                ', ],
            'pbp_cart_bundle' => [
                'ddl' => '
                    CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . "ppbs_product` (
                      `id_product` mediumint unsigned NOT NULL,
                      `id_unit` smallint unsigned NOT NULL DEFAULT '0',
                      `is_enabled` tinyint unsigned NOT NULL DEFAULT '0',
                      `type` enum('ppbs','pap') COLLATE utf8mb4_general_ci DEFAULT 'ppbs',
                      `is_front_conversion_enabled` tinyint unsigned NOT NULL DEFAULT '0',
                      `front_conversion_operator` varchar(3) COLLATE utf8mb4_general_ci DEFAULT NULL,
                      `front_conversion_value` decimal(8,2) unsigned DEFAULT '0.00',
                      `min_total_area` decimal(8,2) unsigned DEFAULT '0.00',
                      `min_price` decimal(8,2) unsigned DEFAULT '0.00',
                      `setup_fee` decimal(8,2) unsigned DEFAULT '0.00',
                      `pap_type` enum('normal','rolls','paint') COLLATE utf8mb4_general_ci DEFAULT NULL,
                      `is_dynamic_price` tinyint unsigned DEFAULT '0',
                      `is_attribute_price_area_price` tinyint unsigned DEFAULT '0',
                      `is_stock_enabled` tinyint unsigned NOT NULL DEFAULT '0',
                      PRIMARY KEY (`id_product`)                    
                    ) ENGINE=" . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8 ;            
                ', ],
            'ppbs_product_equation' => [
                'ddl' => '
                    CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . "ppbs_product_equation` (
                      `id_product_equation` mediumint unsigned NOT NULL AUTO_INCREMENT,
                      `id_product` mediumint unsigned NOT NULL DEFAULT '0',
                      `id_product_attribute` mediumint unsigned NOT NULL DEFAULT '0',
                      `id_equation_template` mediumint unsigned NOT NULL DEFAULT '0',
                      PRIMARY KEY (`id_product_equation`)                    
                    ) ENGINE=" . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8 ;            
                ', ],
            'ppbs_product_field' => [
                'ddl' => '
                    CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . "ppbs_product_field` (
                      `id_product_field` mediumint unsigned NOT NULL AUTO_INCREMENT,
                      `id_product` mediumint unsigned NOT NULL,
                      `id_dimension` smallint unsigned NOT NULL,
                      `id_field_type` smallint unsigned NOT NULL,
                      `id_unit` smallint unsigned NOT NULL DEFAULT '0',
                      `min` decimal(8,2) unsigned NOT NULL DEFAULT '0.00',
                      `max` decimal(8,2) unsigned NOT NULL DEFAULT '0.00',
                      `default_value` decimal(8,2) unsigned DEFAULT NULL,
                      `ratio` decimal(8,2) unsigned NOT NULL DEFAULT '0.00',
                      `step` decimal(8,2) unsigned NOT NULL DEFAULT '0.00',
                      `decimals` smallint unsigned NOT NULL DEFAULT '0',
                      `position` smallint unsigned NOT NULL DEFAULT '0',
                      PRIMARY KEY (`id_product_field`)                    
                    ) ENGINE=" . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8 ;            
                ', ],
            'ppbs_product_field_value' => [
                'ddl' => '
                    CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . "ppbs_product_field_value` (
                      `id_product_field_value` mediumint unsigned NOT NULL AUTO_INCREMENT,
                      `id_product_field` smallint unsigned DEFAULT '0',
                      `value` decimal(10,0) unsigned DEFAULT '0',
                      `position` smallint unsigned DEFAULT '0',
                      PRIMARY KEY (`id_product_field_value`)                    
                    ) ENGINE=" . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8 ;            
                ', ],
            'ppbs_product_field_value_lang' => [
                'ddl' => '
                    CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . 'ppbs_product_field_value_lang` (
                      `id_product_field_value` mediumint unsigned NOT NULL,
                      `id_lang` smallint unsigned NOT NULL,
                      `text` varchar(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
                      PRIMARY KEY (`id_product_field_value`,`id_lang`)                    
                    ) ENGINE=' . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8 ;            
                ', ],
            'ppbs_product_pack' => [
                'ddl' => '
                    CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . "ppbs_product_pack` (
                      `id_product_pack` mediumint unsigned NOT NULL AUTO_INCREMENT,
                      `id_product` mediumint unsigned NOT NULL DEFAULT '0',
                      `id_product_attribute` mediumint unsigned NOT NULL DEFAULT '0',
                      `pack_area` decimal(8,2) unsigned NOT NULL DEFAULT '0.00',
                      `area_price` decimal(8,2) unsigned NOT NULL DEFAULT '0.00',
                      `roll_height` decimal(8,2) unsigned NOT NULL DEFAULT '0.00',
                      `roll_width` decimal(8,2) NOT NULL,
                      `pattern_repeat` decimal(8,2) unsigned NOT NULL DEFAULT '0.00',
                      `coverage` decimal(8,2) unsigned NOT NULL DEFAULT '0.00',
                      `type` enum('normal','paint','rolls') CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci NOT NULL DEFAULT 'normal',
                      `is_dynamic_price` tinyint unsigned NOT NULL DEFAULT '0',
                      PRIMARY KEY (`id_product_pack`) USING BTREE                    
                    ) ENGINE=" . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8 ;            
                ', ],
            'ppbs_product_unit' => [
                'ddl' => '
                    CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . "ppbs_product_unit` (
                      `id_product_unit` mediumint unsigned NOT NULL AUTO_INCREMENT,
                      `id_product` mediumint unsigned NOT NULL DEFAULT '0',
                      `id_unit` smallint unsigned NOT NULL DEFAULT '0',
                      `is_default` tinyint unsigned NOT NULL DEFAULT '0',
                      PRIMARY KEY (`id_product_unit`)                    
                    ) ENGINE=" . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8 ;            
                ', ],
            'ppbs_product_wastage' => [
                'ddl' => '
                    CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . "ppbs_product_wastage` (
                      `id_product` mediumint unsigned NOT NULL DEFAULT '0',
                      `id_wastage` smallint unsigned NOT NULL DEFAULT '0',
                      PRIMARY KEY (`id_product`,`id_wastage`)                    
                    ) ENGINE=" . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8 ;            
                ', ],
            'ppbs_stock' => [
                'ddl' => '
                    CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . "ppbs_stock` (
                      `id_stock` mediumint unsigned NOT NULL AUTO_INCREMENT,
                      `id_product` mediumint unsigned NOT NULL DEFAULT '0',
                      `id_product_attribute` mediumint unsigned NOT NULL DEFAULT '0',
                      `id_shop` smallint unsigned NOT NULL DEFAULT '0',
                      `stock` decimal(8,2) unsigned DEFAULT NULL,
                      PRIMARY KEY (`id_stock`)                    
                    ) ENGINE=" . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8 ;            
                ', ],
            'ppbs_unit' => [
                'ddl' => '
                    CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . "ppbs_unit` (
                      `id_unit` smallint unsigned NOT NULL AUTO_INCREMENT,
                      `name` varchar(64) COLLATE utf8mb4_general_ci DEFAULT NULL,
                      `conversion_factor` decimal(8,2) unsigned NOT NULL DEFAULT '0.00',
                      `is_deleted` tinyint unsigned NOT NULL DEFAULT '0',
                      PRIMARY KEY (`id_unit`)                    
                    ) ENGINE=" . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8 ;            
                ', ],
            'ppbs_unit_lang' => [
                'ddl' => '
                    CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . 'ppbs_unit_lang` (
                      `id_unit` smallint unsigned NOT NULL,
                      `id_lang` smallint unsigned NOT NULL,
                      `display_name` varchar(32) COLLATE utf8mb4_general_ci DEFAULT NULL,
                      PRIMARY KEY (`id_unit`,`id_lang`)                    
                    ) ENGINE=' . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8 ;            
                ', ],
            'ppbs_wastage' => [
                'ddl' => '
                    CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . "ppbs_wastage` (
                      `id_wastage` smallint unsigned NOT NULL AUTO_INCREMENT,
                      `value` smallint unsigned NOT NULL DEFAULT '0',
                      PRIMARY KEY (`id_wastage`)                    
                    ) ENGINE=" . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8 ;            
                ', ],
        ];
    }

    public function installSchema(): bool
    {
        foreach ($this->schema as $key => $entry) {
            \Db::getInstance()->execute($entry['ddl']);
        }

        return true;
    }

    public static function addColumn($table, $name, $type): void
    {
        try {
            $return = \Db::getInstance()->execute('ALTER TABLE  `' . _DB_PREFIX_ . bqSQL($table) . '` ADD  `' . bqSQL($name) . '` ' . bqSQL($type));
        } catch (\Exception $e) {
        }
    }

    public function installData(): bool
    {
        $languages = \Context::getContext()->language->getLanguages();

        foreach ($this->data['dimensions'] as $dimension) {
            $inserted = \Db::getInstance()->insert('ppbs_dimension', [
                'name' => $dimension['name'],
                'is_deleted' => 0,
            ]);

            if ($inserted) {
                $idDimension = \Db::getInstance()->Insert_ID();
                foreach ($languages as $lang) {
                    \Db::getInstance()->insert('ppbs_dimension_lang', [
                        'id_dimension' => $idDimension,
                        'id_lang' => $lang['id_lang'],
                        'display_name' => $dimension['lang']['text'],
                    ]);
                }
            }
        }

        foreach ($this->data['units'] as $unit) {
            $inserted = \Db::getInstance()->insert('ppbs_unit', [
                'name' => $unit['name'],
                'conversion_factor' => $unit['conversion_factor'],
                'is_deleted' => 0,
            ]);

            if ($inserted) {
                $idUnit = \Db::getInstance()->Insert_ID();
                foreach ($languages as $lang) {
                    \Db::getInstance()->insert('ppbs_unit_lang', [
                        'id_unit' => $idUnit,
                        'id_lang' => $lang['id_lang'],
                        'display_name' => $unit['lang']['text'],
                    ]);
                }
            }
        }

        foreach ($this->data['field_type'] as $fieldType) {
            \Db::getInstance()->insert('ppbs_field_type', [
                'name' => pSQL($fieldType),
            ]);
        }

        foreach ($this->data['equation_type'] as $equationType) {
            \Db::getInstance()->insert('ppbs_equation_type', [
                'name' => pSQL($equationType['name']),
            ]);
        }

        self::addColumn('customized_data', 'ppbs_data', 'TEXT');
        self::addColumn('customization_field', 'is_ppbs', 'tinyint(1) UNSIGNED NOT NULL');
        return true;
    }

    public function removeIndexFiles($folder, $isRoot = true)
    {
        $files = scandir($folder);
        foreach ($files as $file) {
            if ($file === '.' || $file === '..') {
                continue;
            }
            $filePath = $folder . DIRECTORY_SEPARATOR . $file;
            if (is_dir($filePath)) {
                self::removeIndexFiles($filePath, false);
            }
            if (!$isRoot && is_file($filePath) && $file === 'index.php') {
                unlink($filePath);
            }
        }
    }
}
