<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    Musaffar Patel
 * @copyright 2016-2025 Musaffar Patel
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 */

namespace PrestaShop\Module\ProductPriceBySize\Service\Common;

use PrestaShop\PrestaShop\Adapter\Product\PriceFormatter;

if (!defined('_PS_VERSION_')) {
    exit;
}

class AbstractPriceService
{
    protected $context;

    protected $moduleInstance;

    protected $productRepository;

    protected $cartRepository;

    protected $areaPriceRepository;

    protected $equationRepository;

    protected $equationVariableRepository;

    protected $dimensionRepository;

    protected $priceFormatter;

    protected static $priceCache;

    public function __construct(\Context $context, \Module $moduleInstance)
    {
        $this->context = $context;
        $this->moduleInstance = $moduleInstance;
        $this->cartRepository = $this->moduleInstance->get('prestashop.module.productpricebysize.repository.cart_repository');
        $this->productRepository = $this->moduleInstance->get('prestashop.module.productpricebysize.repository.product_repository');
        $this->areaPriceRepository = $this->moduleInstance->get('prestashop.module.productpricebysize.repository.area_price_repository');
        $this->equationRepository = $this->moduleInstance->get('prestashop.module.productpricebysize.repository.equation_repository');
        $this->equationVariableRepository = $this->moduleInstance->get('prestashop.module.productpricebysize.repository.equation_variable_repository');
        $this->dimensionRepository = $this->moduleInstance->get('prestashop.module.productpricebysize.repository.dimension_repository');
        $this->priceFormatter = new PriceFormatter();
    }

    public static function addTax($taxManager, $use_tax, $price)
    {
        if ($use_tax) {
            return $taxManager->addTaxes($price);
        } else {
            return $price;
        }
    }

    protected function getProductAttributePrice(int $idProduct, int $idProductAttribute): array
    {
        $cacheId2 = $idProduct . '-' . $this->context->shop->id;
        if (empty(self::$priceCache[$cacheId2])) {
            $attributePriceData = $this->productRepository->getAttributePriceData($idProduct, $this->context->shop->id);
            foreach ($attributePriceData as $row) {
                $arrayTmp = [
                    'price' => $row['price'],
                    'ecotax' => $row['ecotax'],
                    'attribute_price' => (isset($row['attribute_price']) ? $row['attribute_price'] : null),
                ];
                self::$priceCache[$cacheId2][(int) $row['id_product_attribute']] = $arrayTmp;
                if (isset($row['default_on']) && $row['default_on'] == 1) {
                    self::$priceCache[$cacheId2][0] = $arrayTmp;
                }
            }
        }

        return self::$priceCache[$cacheId2][$idProductAttribute];
    }

    protected function getCustomerGroupReduction(\Customer $customer): float
    {
        $groupReduction = 0;
        if (!empty($customer->id_default_group)) {
            $idGroup = $customer->id_default_group;
            $customerGroup = new \Group($idGroup);
            $groupReduction = $customerGroup->reduction;
        }

        return $groupReduction;
    }

    protected function getProductTaxCalculation(?\Address $address, array $params): \TaxCalculator
    {
        if (empty($address)) {
            $address = new \Address();
        }
        $address->id_country = $params['id_country'];
        $address->id_state = $params['id_state'];
        $address->postcode = $params['zip_code'];
        $tax_manager = \TaxManagerFactory::getManager($address, \Product::getIdTaxRulesGroupByIdProduct((int) $params['id_product'], $this->context));

        return $tax_manager->getTaxCalculator();
    }

    protected static function applyDiscount($price, $amount, $type): float
    {
        if ($type == 'percentage') {
            $price = $price - ($price * ($amount / 100));
        }

        if ($type == 'amount') {
            $price = $price - $amount;
        }

        return $price;
    }

    protected function applySpecificPrice(float $originalPrice, array $params, \TaxCalculator $productTaxCalculator, bool $isAttribute): float
    {
        $newPrice = $originalPrice;
        if (isset($params['specific_price']['price'])) {
            $specificPrice = $params['specific_price'];
            $price = $specificPrice['price'];
            if (isset($specificPrice['reduction']) && isset($specificPrice['reduction_type'])) {
                $reduction = $specificPrice['reduction'];

                if ($specificPrice['reduction_type'] == 'amount') {
                    if ($specificPrice['reduction_tax'] == 1) {
                        $reduction = $productTaxCalculator->removeTaxes($specificPrice['reduction']);
                    } else {
                        $reduction = $specificPrice['reduction'];
                    }
                }

                if ($reduction == 0 && $price > 0) {
                    $newPrice = $price;
                    if ($isAttribute) {
                        $newPrice = 0;
                    }
                } else {
                    if ($specificPrice['reduction_type'] == 'percentage' && $reduction < 1) {
                        $reduction = $reduction * 100;
                    } elseif ($specificPrice['reduction_type'] == 'amount') {
                    }
                    $newPrice = $this->applyDiscount($originalPrice, $reduction, $specificPrice['reduction_type']);
                }
            }
        }

        return $newPrice;
    }

    public function applyAreaPriceImpact(float $originalPrice, float $areaPrice, string $operation): float
    {
        switch ($operation) {
            case '-':
                return $originalPrice - $areaPrice;
            case '+':
                return $originalPrice + $areaPrice;
            case '*':
                return $originalPrice * $areaPrice;
            case '=':
            case '~':
                return $areaPrice;
            default:
                return 0;
        }
    }
}
