<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    Musaffar Patel
 * @copyright 2016-2025 Musaffar Patel
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 */

namespace PrestaShop\Module\ProductPriceBySize\Service\Ppbs;

use PrestaShop\Module\ProductPriceBySize\Entity\PpbsProduct;
use PrestaShop\Module\ProductPriceBySize\Enum\EquationType;
use PrestaShop\Module\ProductPriceBySize\Service\Common\AbstractCartService;
use PrestaShop\Module\ProductPriceBySize\Service\FormatterService;
use PrestaShop\Module\ProductPriceBySize\Service\MathParser\Parser;

if (!defined('_PS_VERSION_')) {
    exit;
}

class CartService extends AbstractCartService
{
    private $productFieldRepository;

    public function __construct(\Context $context, \Module $moduleInstance)
    {
        parent::__construct($context, $moduleInstance);
        $this->productFieldRepository = $this->moduleInstance->get('prestashop.module.productpricebysize.repository.product_field_repository');
    }

    public function constructCartData(array $formData): array
    {
        $cartData = [
            'idProduct' => (int) $formData['idProduct'],
            'fields' => [],
        ];

        $productFields = $this->productFieldRepository->getByProduct((int) $formData['idProduct'], $this->context->language->id);
        foreach ($productFields as $productField) {
            $productField = $this->productFieldRepository->getOneByLang((int) $productField['id'], $this->context->language->id);
            $value = 0;
            foreach ($formData['fields'] as $formField) {
                if ((int) $formField['idProductField'] === (int) $productField['id']) {
                    $value = $formField['value'];
                    break;
                }
            }

            if (!empty($productField)) {
                $cartData['fields'][] = [
                    'idProductField' => $productField['id'],
                    'value' => $value,
                    'dimension' => [
                        'id' => $productField['dimension']['id'],
                        'displayName' => $productField['dimension']['lang']['displayName'],
                    ],
                    'unit' => [
                        'id' => $productField['unit']['id'],
                        'displayName' => $productField['unit']['lang']['displayName'],
                    ],
                ];
            }
        }

        return $cartData;
    }

    public function addToCart(int $idProduct, $idCustomization, $group, int $ipa, int $qty, string $formData): int
    {
        $formData = json_decode($formData, true);

        if (empty($formData)) {
            return 0;
        }

        $this->checkAndCreateCart();
        if (!isset($this->context->cart->id)) {
            return 0;
        }

        $idProductAttribute = $this->getIdProductAttributeFromGroup($idProduct, $group, $ipa);
        if ($qty == 0) {
            $qty = 1;
        }

        $cartData = $this->constructCartData($formData);

        if ($idCustomization > 0) {
            $customizedDataText = $this->cartRepository->getCustomizationDisplayText($cartData);
            $idCustomizationField = $this->cartRepository->getCustomizationField($idProduct, $this->context->shop->id);
            $this->cartRepository->addCustomizedData($idCustomization, $idCustomizationField, $customizedDataText, (int) $this->moduleInstance->id, $cartData);
        } else {
            $idCustomization = $this->cartRepository->addCustomization(
                $idProduct,
                $this->context->cart->id,
                $idProductAttribute,
                \Context::getContext()->cart->id_address_delivery,
                $cartData,
                $this->moduleInstance->id,
                $qty,
                $this->context->shop->id);
        }

        return $idCustomization;
    }

    public function getTotalArea($cartDataRow, PpbsProduct $ppbsProduct): float
    {
        $multiplier = 1;
        $totalArea = 0;

        foreach ($cartDataRow['productFields']['fields'] as $productField) {
            $productField['value'] = FormatterService::removeNumberFormatting($productField['value']);

            if ((float) $productField['value'] > 0) {
                if ($ppbsProduct->getIsFrontConversionEnabled()) {
                    switch ($ppbsProduct->getFrontConversionOperator()) {
                        case '/':
                            if ($ppbsProduct->getFrontConversionValue() == 0) {
                                $ppbsProduct->setFrontConversionValue(1);
                            }
                            $productField['value'] = $productField['value'] / $ppbsProduct->getFrontConversionValue();
                            break;
                        case '*':
                            if ($ppbsProduct->getFrontConversionValue() == 0) {
                                $ppbsProduct->setFrontConversionValue(1);
                            }
                            $productField = $productField * $ppbsProduct->getFrontConversionValue();
                            break;
                    }
                }
                $totalArea = $totalArea * $productField['value'];
                $multiplier = $multiplier * $productField['value'];
            }
            if ($totalArea == 0) {
                $totalArea = $productField['value'];
            }
        }

        return $totalArea;
    }

    private function calculateEquation(string $equation, array $mathParams): float
    {
        return Parser::computeEquation($equation, $mathParams);
    }

    public function getTotalWeight(array $products, int $idCart, bool $withQty): float
    {
        $totalWeight = 0;

        foreach ($products as $product) {
            $factory = $this->moduleInstance->get('prestashop.module.productpricebysize.factory.price_service_factory');
            $priceService = $factory->get((int) $product['id_product']);
            if (empty($priceService)) {
                continue;
            }

            $idProduct = $product['id_product'];
            $idProductAttribute = $product['id_product_attribute'];
            $idCustomization = $product['id_customization'];
            $ppbsProduct = $this->productRepository->find($idProduct);
            $cartData = $this->cartRepository->getCartData($idProduct, $idCart, $idProductAttribute, (int) $this->context->shop->id, $idCustomization);

            $productTotalArea = 0;

            foreach ($cartData as $cartDataRow) {
                $productTotalArea += $this->getTotalArea($cartDataRow, $ppbsProduct);
            }

            $mathParams = [];
            $mathParams['total_area'] = $productTotalArea;
            $mathParams['product_weight'] = (float) $product['weight'];

            $equationEntity = $this->equationRepository->getByProduct($idProduct, 0, EquationType::WEIGHT);

            if (empty($equationEntity['equation']['equation'])) {
                $equation = '[total_area]*[product_weight]';
            } else {
                $equation = $equationEntity['equation']['equation'];
            }

            $this->equationService->populateMathParams($mathParams, $cartData, $idProduct, $idProductAttribute, $idCustomization, $idCart);
            $totalWeight += $this->calculateEquation($equation, $mathParams);
        }

        return $totalWeight;
    }
}
