<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    Musaffar Patel
 * @copyright 2016-2025 Musaffar Patel
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 */

namespace PrestaShop\Module\ProductPriceBySize\Service\Ppbs;

use PrestaShop\Module\ProductPriceBySize\Entity\PpbsProduct;
use PrestaShop\Module\ProductPriceBySize\Enum\EquationType;
use PrestaShop\Module\ProductPriceBySize\Service\Common\AbstractPriceService;
use PrestaShop\Module\ProductPriceBySize\Service\Common\PriceServiceInterface;
use PrestaShop\Module\ProductPriceBySize\Service\FormatterService;
use PrestaShop\Module\ProductPriceBySize\Service\MathParser\Parser;
use PrestaShop\Module\ProductPriceBySize\Service\NumberService;

if (!defined('_PS_VERSION_')) {
    exit;
}

class PriceService extends AbstractPriceService implements PriceServiceInterface
{
    protected $cartService;

    public function __construct(\Context $context, \Module $moduleInstance)
    {
        parent::__construct($context, $moduleInstance);
        $this->cartService = new CartService($context, $moduleInstance);
    }

    public function customEquationCalculation(array $params,
        array $cartData,
        PpbsProduct $ppbsProduct,
        int $idProduct,
        float $priceBase,
        float $priceAttribute,
        $productTaxCalculator,
        float $customerGroupReduction,
        string $equation,
        bool $isCustomizationLine): float
    {
        $runningTotal = 0;

        foreach ($cartData as $cartDataRow) {
            $totalArea = $this->cartService->getTotalArea($cartDataRow, $ppbsProduct);
            $areaPrice = $this->areaPriceRepository->getByArea($idProduct, $totalArea, (int) $this->context->shop->id);
            if ($totalArea < $ppbsProduct->getMinTotalarea() && $ppbsProduct->getMinTotalArea() > 0) {
                $totalArea = $ppbsProduct->getMinTotalArea();
            }

            $mathParams = [];
            $mathParams['product_price'] = $priceBase + $priceAttribute;
            $mathParams['base_price'] = $priceBase;
            $mathParams['attribute_price'] = $priceAttribute;
            $mathParams['quantity'] = (int) $params['quantity'];
            $mathParams['area_price'] = !empty($areaPrice['price']) ? (float) $areaPrice['price'] : 0;
            $mathParams['total_area'] = $totalArea;
            $mathParams['pco_price_impact'] = !empty($params['pco_price_impact']) ? (float) $params['pco_price_impact'] : 0;

            $globalMathVariables = $this->equationVariableRepository->findAll();
            foreach ($globalMathVariables as $globalMathVariable) {
                $mathParams[$globalMathVariable->getName()] = (float) $globalMathVariable->getValue();
            }

            foreach ($cartData as $cartDataTemp) {
                foreach ($cartDataTemp['productFields']['fields'] as $productField) {
                    $dimension = $this->dimensionRepository->getOne((int) $productField['dimension']['id']);
                    $mathParams[$dimension['name']] = (float) FormatterService::removeNumberFormatting($productField['value']);
                }
            }

            $unitTotal = Parser::computeEquation($equation, $mathParams);

            if ($customerGroupReduction > 0) {
                $unitTotal = $this->applyDiscount($unitTotal, $customerGroupReduction, 'percentage');
            }
            $unitTotal = $this->applySpecificPrice($unitTotal, $params, $productTaxCalculator, false);
            $runningTotal += $unitTotal;
        }
        $runningTotal = $runningTotal + $ppbsProduct->getSetupFee();

        // Impose min price if set
        if ($runningTotal < $productTaxCalculator->addTaxes($ppbsProduct->getMinPrice(), $isCustomizationLine)) {
            $total = $this->addTax($productTaxCalculator, $params['use_tax'], $ppbsProduct->getMinPrice());
        } else {
            $total = $this->addTax($productTaxCalculator, $params['use_tax'], $runningTotal);
        }

        return $this->priceFormatter->convertAmount((float) $total);
    }

    public function normalCalculation(array $params,
        array $cartData,
        PpbsProduct $ppbsProduct,
        int $idProduct,
        float $priceBase,
        float $priceAttribute,
        $productTaxCalculator,
        float $customerGroupReduction): float
    {
        $unitTotal = 0;
        foreach ($cartData as $cartDataRow) {
            $totalArea = $this->cartService->getTotalArea($cartDataRow, $ppbsProduct);
            $totalAreaQty = $totalArea * $params['quantity'];

            if ($totalArea < $ppbsProduct->getMinTotalarea() && $ppbsProduct->getMinTotalArea() > 0) {
                $totalArea = $ppbsProduct->getMinTotalArea();
            }

            $multiplier = $totalArea;
            $priceBase = $this->applySpecificPrice($priceBase, $params, $productTaxCalculator, false);
            $priceAttribute = $this->applySpecificPrice($priceAttribute, $params, $productTaxCalculator, true);

            /* use base price to calculate area, and add attributes price to that total */
            if (!$ppbsProduct->getIsAttributePriceAreaPrice()) {
                $adjustedPrice = $priceBase;
            } else {
                $adjustedPrice = $priceBase + $priceAttribute;
            }

            // Adjust price based on area
            if ($totalAreaQty > 0) {
                $totalAreaQty = NumberService::roundUp($totalAreaQty, 2);
                $areaPrice = $this->areaPriceRepository->getByArea($idProduct, $totalAreaQty, (int) $this->context->shop->id);
                if (!empty($areaPrice)) {
                    $adjustedPrice = $this->applyAreaPriceImpact($adjustedPrice, $areaPrice['price'], $areaPrice['impact']);
                    $adjustedPrice = $this->applySpecificPrice($adjustedPrice, $params, $productTaxCalculator, true);
                }
            }

            if (!empty($areaPrice['impact']) && $areaPrice['impact'] == '=') {
                $unitTotal += ($adjustedPrice + $priceAttribute);
            } else {
                $customisationTotal = ($adjustedPrice * $multiplier);
                if ($customisationTotal < $ppbsProduct->getMinPrice()) {
                    $unitTotal += $ppbsProduct->getMinPrice();
                } else {
                    $unitTotal += ($adjustedPrice * $multiplier);
                }
            }

            if (!empty($params['pco_price_impact'])) {
                $unitTotal += (float) $params['pco_price_impact'];
            }

            $unitTotal = $this->priceFormatter->convertAmount($unitTotal);
            if ($customerGroupReduction > 0) {
                $unitTotal = $this->applyDiscount($unitTotal, $customerGroupReduction, 'percentage');
            }

            if ($ppbsProduct->getSetupFee() > 0) {
                $unitTotal = $unitTotal + $ppbsProduct->getSetupFee();
            }

            // Impose min price if set
            if ($unitTotal < $ppbsProduct->getMinPrice()) {
                $unitTotal = $ppbsProduct->getMinPrice();
                $unitTotal = $this->addTax($productTaxCalculator, $params['use_tax'], $unitTotal);
            } else {
                $unitTotal = $this->addTax($productTaxCalculator, $params['use_tax'], $unitTotal);
            }
        }

        return $unitTotal;
    }

    public function priceCalculation(array $params, ?\Customer $customer, bool $isCustomizationLine, bool $useCache): float
    {
        static $address = null;
        if (empty($customer)) {
            $customer = $this->context->customer;
        }
        $ppbsProduct = $this->productRepository->find($params['id_product']);
        $originalPrice = $params['price'];
        if (empty($this->context->customer) || empty($params['id_product']) || empty($params['id_customization'])) {
            return $originalPrice;
        }

        $cacheId = 'PpbsPriceService::priceCalculation_' . $params['id_product'] . '-' . $params['id_product_attribute'] . '-' . $params['id_customization'] . '-' . $params['id_cart'] . '-' . (int) $params['use_tax'] . ' -' . (int) $params['use_reduc'];

        if ($useCache) {
            if (\Cache::isStored($cacheId)) {
                return \Tools::ps_round(\Cache::retrieve($cacheId), 6);
            }
        }

        $cartData = $this->cartRepository->getCartData((int) $params['id_product'], (int) $params['id_cart'], (int) $params['id_product_attribute'], (int) $this->context->shop->id, (int) $params['id_customization']);
        if (empty($cartData)) {
            return (float) $params['price'];
        }

        $idProduct = (int) $params['id_product'];
        $product = new \Product($idProduct, null, null, $this->context->shop->id);
        $priceAttribute = (int) $this->getProductAttributePrice($idProduct, $params['id_product_attribute'])['attribute_price'];
        $customerGroupReduction = $this->getCustomerGroupReduction($customer);
        $productTaxCalculator = $this->getProductTaxCalculation($address, $params);
        $priceBase = $product->price;
        $equation = $this->equationRepository->getByProduct($idProduct, 0, EquationType::PRICE);

        if (!empty($equation)) {
            $price = $this->customEquationCalculation($params, $cartData, $ppbsProduct, $idProduct, $priceBase, $priceAttribute, $productTaxCalculator, $customerGroupReduction, $equation['equationTemplate']['equation'], $isCustomizationLine);
        } else {
            $price = $this->normalCalculation($params, $cartData, $ppbsProduct, $idProduct, $priceBase, $priceAttribute, $productTaxCalculator, $customerGroupReduction);
        }

        \Cache::store($cacheId, $price);

        return \Tools::ps_round($price, 6);
    }
}
